﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Linq;
using System.Text;
using System.Diagnostics;
using System.Windows.Forms;
using System.IO;

using System.Threading;
using System.Threading.Tasks;

using Agilent.AgM9391.Interop;
using Agilent.AgM938x.Interop;


namespace NSM9391A
{
    public partial class Vsag_M9391A
    {
        // Properties that will modify measurement procedures
        public bool useChanPwrForRef = true;
        public bool useVsaOffsetFreq = true;
        double fStart = 0;
        double fDelta = 0;

        #region Parameter Setup
        public void setupParameters(string condition)
        {
            switch (condition.ToUpper())
            {
                case "CW":
                    // ACPR Offsets
                    // Acquisition Settings
                    ifBandwidth = 40e6;
                    sampleRate = 1e6;
                    pwrDuration = 10e-6;
                    acprDuration = pwrDuration;
                    FilterType = AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeRectangular;
                    FilterAlpha = 0.3;
                    FilterBw = 800e3;
                    refLevelMargin = 5;
                    useVsaOffsetFreq = true;
                    break;
                case "GSM":
                case "EDGE":
                    // ACPR Offsets
                    numAcprMeas = 4;
                    acprOffsetFreq[0] = -400e3;
                    acprOffsetFreq[1] = 400e3;
                    acprOffsetFreq[2] = -600e3;
                    acprOffsetFreq[3] = 600e3;
                    // Acquisition Settings
                    ifBandwidth = 40e6;
                    sampleRate = 1.25e6;
                    pwrDuration = 1040e-6;      //two time slots
                    acprDuration = pwrDuration;
                    FilterType = AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeGaussian;
                    FilterAlpha = 0.3;
                    FilterBw = 271e3;
                    acprFilterBw = 30e3;
                    acprSpan = 1.667e6 / 1.25;
                    refLevelMargin = 5;
                    useChanPwrForRef = true;
                    useVsaOffsetFreq = true;
                    break;
                case "CDMA2000":
                case "EVDO":
                    // ACPR Offsets
                    numAcprMeas = 4;
                    acprOffsetFreq[0] = -1.98e6;
                    acprOffsetFreq[1] = -.75e6;
                    acprOffsetFreq[2] = .75e6;
                    acprOffsetFreq[3] = 1.98e6;
                    // Acquisition Settings
                    ifBandwidth = 15e6;
                    sampleRate = 2.5e6;
                    pwrDuration = 200e-6;
                    acprDuration = 2500e-6;
                    FilterType = AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeRectangular;
                    FilterAlpha = 0.22;
                    FilterBw = 1.23e6;
                    acprFilterBw = 30e3;
                    acprFilterType = AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeRectangular;
                    acprFilterAlpha = 0.22;
                    acprSpan = 5e6;
                    refLevelMargin = 5;
                    break;
                case "WCDMA":
                    // ACPR Offsets
                    numAcprMeas = 4;
                    acprOffsetFreq[0] = -5e6;
                    acprOffsetFreq[1] = 5e6;
                    acprOffsetFreq[2] = -10e6;
                    acprOffsetFreq[3] = 10e6;
                    // Acquisition Settings
                    ifBandwidth = 40e6;
                    sampleRate = 5e6;
                    pwrDuration = 100e-6;
                    acprDuration = 500e-6;
                    FilterType = AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeRootRaisedCosine;
                    FilterAlpha = 0.22;
                    FilterBw = 3840000.0;
                    acprFilterType = AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeRootRaisedCosine;
                    acprFilterBw = 3.84e6;
                    acprFilterAlpha = 0.22;
                    acprSpan = 30.72e6 / 1.25; //25e6;
                    refLevelMargin = 5;
                    break;
                case "TD-SCDMA":
                    // ACPR Offsets
                    numAcprMeas = 4;
                    acprOffsetFreq[0] = -1.6e6;
                    acprOffsetFreq[1] = 1.6e6;
                    acprOffsetFreq[2] = -3.2e6;
                    acprOffsetFreq[3] = 3.2e6;
                    // Acquisition Settings
                    ifBandwidth = 40e6;
                    sampleRate = 5e6;
                    pwrDuration = 1000e-6;
                    acprDuration = 1000e-6;
                    FilterType = AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeRootRaisedCosine;
                    FilterAlpha = 0.22;
                    FilterBw = 1.28e6;
                    acprFilterType = AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeRootRaisedCosine;
                    acprFilterBw = 1.28e6;
                    acprFilterAlpha = 0.22;
                    acprSpan = 12.5e6 / 1.25; //25e6;
                    refLevelMargin = 5;
                    break;
                case "LTE_FDD_5MHZ":
                    // ACPR Offsets
                    numAcprMeas = 6;
                    acprOffsetFreq[0] = -5e6;
                    acprOffsetFreq[1] = 5e6;
                    acprOffsetFreq[2] = -10e6;
                    acprOffsetFreq[3] = 10e6;
                    acprOffsetFreq[4] = -5e6;
                    acprOffsetFreq[5] = 5e6;
                    // Channel and Filter settings
                    ifBandwidth = 40e6;
                    sampleRate = 5.625e6;
                    FilterType = AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeRectangular;
                    FilterAlpha = 0.22;
                    FilterBw = 4.5e6;
                    acprFilterType = AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeRootRaisedCosine;
                    acprFilterBw = 3.84e6;
                    acprSpan = 38.4e6 / 1.25;//30e6;
                    pwrDuration = 500e-6;
                    acprDuration = 500e-6;
                    refLevelMargin = 5;
                    break;
                case "LTE_TDD_5MHZ":
                    // ACPR Offsets
                    numAcprMeas = 6;
                    acprOffsetFreq[0] = -3.3e6;
                    acprOffsetFreq[1] = 3.3e6;
                    acprOffsetFreq[2] = -4.9e6;
                    acprOffsetFreq[3] = 4.9e6;
                    acprOffsetFreq[4] = -5e6;
                    acprOffsetFreq[5] = 5e6;
                    // Channel and Filter settings
                    ifBandwidth = 40e6;
                    sampleRate = 5.625e6;
                    FilterType = AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeRectangular;
                    FilterAlpha = 0.22;
                    FilterBw = 4.5e6;
                    acprFilterType = AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeRootRaisedCosine;
                    acprFilterBw = 1.28e6;
                    acprSpan = 38.4e6 / 1.25;//30e6;
                    pwrDuration = 100e-6;
                    acprDuration = 500e-6;
                    refLevelMargin = 5;
                    break;
                case "LTE_FDD_10MHZ":
                    // ACPR Offsets
                    numAcprMeas = 6;
                    acprOffsetFreq[0] = -7.5e6;
                    acprOffsetFreq[1] = 7.5e6;
                    acprOffsetFreq[2] = -12.5e6;
                    acprOffsetFreq[3] = 12.5e6;
                    acprOffsetFreq[4] = -10e6;
                    acprOffsetFreq[5] = 10e6;
                    // Channel and Filter settings
                    ifBandwidth = 40e6;
                    sampleRate = 11.25e6;
                    pwrDuration = 100e-6;
                    acprDuration = 600e-6;
                    FilterType = AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeRectangular;
                    FilterAlpha = 0.22;
                    FilterBw = 9e6;
                    acprFilterType = AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeRootRaisedCosine;
                    acprFilterBw = 3.84e6;
                    acprSpan = 38.4e6 / 1.25;//30e6;
                    refLevelMargin = 5;
                    break;
                case "LTE_TDD_10MHZ":
                    // ACPR Offsets
                    numAcprMeas = 6;
                    acprOffsetFreq[0] = -5.8e6;
                    acprOffsetFreq[1] = 5.8e6;
                    acprOffsetFreq[2] = -7.4e6;
                    acprOffsetFreq[3] = 7.4e6;
                    acprOffsetFreq[4] = -10e6;
                    acprOffsetFreq[5] = 10e6;
                    // Channel and Filter settings
                    ifBandwidth = 40e6;
                    sampleRate = 11.25e6;
                    pwrDuration = 100e-6;
                    acprDuration = 500e-6;
                    FilterType = AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeRectangular;
                    FilterAlpha = 0.22;
                    FilterBw = 9e6;
                    acprFilterType = AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeRootRaisedCosine;
                    acprFilterBw = 1.28e6;
                    acprSpan = 38.4e6 / 1.25;//30e6;
                    refLevelMargin = 5;
                    break;
                case "LTE_FDD_20MHZ":
                    // ACPR Offsets
                    numAcprMeas = 6;
                    acprOffsetFreq[0] = -12.5e6;
                    acprOffsetFreq[1] = 12.5e6;
                    acprOffsetFreq[2] = -17.5e6;
                    acprOffsetFreq[3] = 17.5e6;
                    acprOffsetFreq[4] = -20e6;
                    acprOffsetFreq[5] = 20e6;
                    // Channel and Filter settings
                    ifBandwidth = 160e6;
                    sampleRate = 22.5e6;
                    pwrDuration = 100e-6;
                    acprDuration = 500e-6;
                    FilterType = AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeRectangular;
                    FilterAlpha = 0.22;
                    FilterBw = 18e6;
                    acprFilterType = AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeRootRaisedCosine;
                    acprFilterBw = 3.84e6;
                    acprSpan = 76.8e6 / 1.25;//60e6;
                    refLevelMargin = 5;
                    break;
                case "LTE_TDD_20MHZ":
                    // ACPR Offsets
                    numAcprMeas = 6;
                    acprOffsetFreq[0] = -10.8e6;
                    acprOffsetFreq[1] = 10.8e6;
                    acprOffsetFreq[2] = -12.4e6;
                    acprOffsetFreq[3] = 12.4e6;
                    acprOffsetFreq[4] = -20e6;
                    acprOffsetFreq[5] = 20e6;
                    // Channel and Filter settings
                    ifBandwidth = 160e6;
                    sampleRate = 22.5e6;
                    pwrDuration = 100e-6;
                    acprDuration = 500e-6;
                    FilterType = AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeRectangular;
                    FilterAlpha = 0.22;
                    FilterBw = 18e6;
                    acprFilterType = AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeRootRaisedCosine;
                    acprFilterBw = 1.28e6;
                    acprSpan = 76.8e6 / 1.25;//60e6;
                    refLevelMargin = 5;
                    break;
                case "WLAN_N_20MHZ":
                    // ACPR Offsets
                    numAcprMeas = 4;
                    acprOffsetFreq[0] = -20e6;
                    acprOffsetFreq[1] = 20e6;
                    acprOffsetFreq[2] = -40e6;
                    acprOffsetFreq[3] = 40e6;
                    // Channel and Filter settings
                    ifBandwidth = 160e6;
                    sampleRate = 25e6;
                    pwrDuration = 16e-6;
                    acprDuration = pwrDuration;
                    acprSpan = 76.8e6 / 1.25;//60e6;
                    FilterType = AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeRectangular;
                    FilterAlpha = 0.22;
                    FilterBw = 20e6;
                    refLevelMargin = 5;
                    break;
                case "WLAN_N_40MHZ":
                case "WLAN_AC_40MHZ":
                    // ACPR Offsets
                    numAcprMeas = 2;
                    acprOffsetFreq[0] = -40e6;
                    acprOffsetFreq[1] = 40e6;
                    acprOffsetFreq[2] = -80e6;
                    acprOffsetFreq[3] = 80e6;
                    // Channel and Filter settings
                    ifBandwidth = 160e6;
                    sampleRate = 50e6;
                    pwrDuration = 16e-6;
                    acprDuration = pwrDuration;
                    acprSpan = 125e6;
                    FilterType = AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeRectangular;
                    FilterAlpha = 0.22;
                    FilterBw = 40e6;
                    refLevelMargin = 5;
                    break;
                case "WLAN_AC_80MHZ":
                    // ACPR Offsets
                    numAcprMeas = 2;
                    acprOffsetFreq[0] = -80e6;
                    acprOffsetFreq[1] = 80e6;
                    acprOffsetFreq[2] = -160e6;
                    acprOffsetFreq[3] = 160e6;
                    // Channel and Filter settings
                    ifBandwidth = 160e6;
                    sampleRate = 100e6;
                    pwrDuration = 16e-6;
                    acprDuration = pwrDuration;
                    acprSpan = 160e6;
                    FilterType = AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeRectangular;
                    FilterAlpha = 0.22;
                    FilterBw = 80e6;
                    refLevelMargin = 5;
                    break;
                default:
                    throw (new Exception("Unrecognized Setup:  " + condition));
            }
        }
        #endregion

        #region Common Setup Routines

        //
        // setupVsgVsa:
        // Configure the VSG and VSA for a power amp test that will use the baseband level property of the VSG to set the exact input
        // level.
        public void setupVsgVsa(string testWaveform, double centerFreq, double targetPout, double targetGain,
            bool useShortWaveform = false, double shortTime = 0, bool isSequence = false)
        {
            //Start Test timer
            sw1.Reset();
            sw1.Start();

            // Start a seperate thread for the VSG setup
            Task vsg = Task.Factory.StartNew(() =>
                {
                    // Setup the VSG
                    double sourceMaxLevel = targetPout - targetGain + getInputAtten(centerFreq, 0) + servoHeadRoom;
                    Vsg.RF.Frequency = centerFreq;
                    Vsg.RF.Level = sourceMaxLevel;
                    Vsg.Modulation.BasebandPower = -1 * servoHeadRoom;

                    if (useShortWaveform)
                    {
                        if (shortTime == 0)
                            shortTime = acprDuration + servoOverheadTime;
                        // Set M9381A Sync Output to ARB Wrap
                        Vsg.Triggers.SynchronizationOutputTrigger.Type = AgM938xSynchronizationTriggerTypeEnum.AgM938xSynchronizationTriggerTypePerArb;
                        Vsg.Modulation2.PlayArbSubset(testWaveform, AgM938xStartEventEnum.AgM938xStartEventImmediate, shortTime);
                    }
                    else if (isSequence)
                    {
                        // Set M9381A Sync Output to Sequence Wrap
                        Vsg.Triggers.SynchronizationOutputTrigger.Type = AgM938xSynchronizationTriggerTypeEnum.AgM938xSynchronizationTriggerTypePerSequence;
                        Vsg.Modulation.Sequence.Play("Sequence", AgM938xStartEventEnum.AgM938xStartEventImmediate);
                    }
                    else
                    {
                        // Set M9381A Sync Output to ARB Wrap
                        Vsg.Triggers.SynchronizationOutputTrigger.Type = AgM938xSynchronizationTriggerTypeEnum.AgM938xSynchronizationTriggerTypePerArb;
                        Vsg.Modulation.PlayArb(testWaveform, AgM938xStartEventEnum.AgM938xStartEventImmediate);
                    }
                }
                );

            // Setup the VSA
            // Get RMS value from playing waveform
            Vsg.Modulation.IQ.ArbInformation(testWaveform, ref arbSampleRate, ref arbRmsValue, arbScale, rfMarker, alcMarker);
            setupVsa(centerFreq, targetPout, false);

            // Wait for the VSG to complete
            vsg.Wait();

            // Log Results
            string[] dataNames = { "setupVsgVsa Time" };
            string[] dataUnits = { "ms" };
            string[] numFormats = { "0" };
            double[] dataValues = { sw1.ElapsedMilliseconds };
            dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
        }

        //
        // setupVsa:
        // Configure the VSA for a power amp test that will use the baseband level property of the VSG to set the exact input
        // level.
        public void setupVsa(double centerFreq, double targetPout, bool logData = true)
        {
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            // Setup the VSA
            // Adjust PLL for best performance for WLAN, GSM and EDGE
            Vsa.Modules.Synthesizer.PLLMode = AgM9391SynthesizerPLLModeEnum.AgM9391SynthesizerPLLModeBestWideOffset;
            Vsa.RF.Frequency = centerFreq;
            Vsa.RF.Power = targetPout + refLevelMargin - getOutputAtten(Vsa.RF.Frequency + Vsa.PowerAcquisition.OffsetFrequency, 0);
            Vsa.RF.Conversion = AgM9391ConversionEnum.AgM9391ConversionAuto;
            Vsa.RF.PeakToAverage = arbRmsValue;
            Vsa.RF.IFBandwidth = ifBandwidth;
            if (fftServo == true)
            {
                // Configure the Acquisition to yuse Real Time FFT Mode
                Vsa.AcquisitionMode = AgM9391AcquisitionModeEnum.AgM9391AcquisitionModeFFT;
                int spectrumCount = (int)(pwrDuration * acprSpan * 1.25 / (int)fftSize);// seconds * samples/second / fftSize
                Vsa.FFTAcquisition.Configure(fftSize, acprSpan * 1.25, spectrumCount, fftWindow, 0);

                // Set the Effective Noise Bandwidth based on Window Type
                // 1.0 for Uniform, 1.5 for Hann, 2.215 for GaussTop, 3.8194 for FlatTop 
                switch (fftWindow)
                {
                    case AgM9391FFTWindowShapeEnum.AgM9391FFTWindowShapeUniform:
                        {
                            windowEnbw = 1.0;
                            break;
                        }
                    case AgM9391FFTWindowShapeEnum.AgM9391FFTWindowShapeHann:
                        {
                            windowEnbw = 1.5;
                            break;
                        }
                    case AgM9391FFTWindowShapeEnum.AgM9391FFTWindowShapeGaussian:
                        {
                            windowEnbw = 2.215;
                            break;
                        }
                    case AgM9391FFTWindowShapeEnum.AgM9391FFTWindowShapeFlatTop:
                        {
                            windowEnbw = 3.8194;
                            break;
                        }
                    default:
                        {
                            throw (new Exception("Window Mode Not Supported:" + fftWindow.ToString()));
                        }
                }
            }
            else
            {
                // Configure the Acquisition
                Vsa.AcquisitionMode = AgM9391AcquisitionModeEnum.AgM9391AcquisitionModePower;
                Vsa.PowerAcquisition.Bandwidth = sampleRate;
                Vsa.PowerAcquisition.Duration = pwrDuration;
                Vsa.PowerAcquisition.ChannelFilter.Configure(FilterType, FilterAlpha, FilterBw);
            }
            // Send Changes to hardware
            Vsa.Apply();

            if (logData)
            {
                // Log Results
                string[] dataNames = { "setupVsa Time" };
                string[] dataUnits = { "ms" };
                string[] numFormats = { "0" };
                double[] dataValues = { sw1.ElapsedMilliseconds };
                dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
            }
        }

        //
        // setTestFreq
        // Sets the Frequency of the VSG using the RF frequency property of the M9381A (Band too wide for offset Frequency)
        // Sets the Frequency of the VSA using the RF frequency (Offset frequency will be used for ACPR offsets)
        public void setTestFreq(double centerFreq)
        {
            //Start Test timer
            sw1.Reset();
            sw1.Start();

            // Start a task for the VSG
            Task vsg = Task.Factory.StartNew(() =>
                {
                    // Set source to Frequency
                    Vsg.RF.Frequency = centerFreq;
                    Vsg.Modulation.BasebandFrequency = 0;
                    if (alignVsg == true && doPowerSearch[dataLog.testIndex] == false)
                    {
                        Vsg.Calibration.PowerSearch.UsePowerSearchResult(rfOffset[dataLog.testIndex], scaleOffset[dataLog.testIndex]);
                    }
                    Vsg.Apply();
                    // Perform the power search at each test on the first DUT run after Init_instruments
                    if (alignVsg == true && doPowerSearch[dataLog.testIndex] == true)
                    {
                        Vsg.Calibration.PowerSearch.BlankRFDuringSearch = true;
                        Vsg.Calibration.PowerSearch.DoPowerSearch(ref rfOffset[dataLog.testIndex], ref scaleOffset[dataLog.testIndex]);
                        doPowerSearch[dataLog.testIndex] = false;
                        if (Math.Abs(rfOffset[dataLog.testIndex]) > 2.0)
                        {
                            Exception ex = new Exception("Rf Offset outside Expected Range: " + rfOffset[dataLog.testIndex].ToString());
                            throw (ex);
                        }
                    }
                }
                );

            // Set VSA to frequency changing the RF frequency
            Vsa.RF.Frequency = centerFreq;
            Vsa.PowerAcquisition.OffsetFrequency = 0;
            Vsa.Apply();
            // Align VSA
            if (alignVsa == true && doVsaAlign[dataLog.testIndex] == true)
            {
                Vsa.Calibration.Amplitude.Align(AgM9391AmplitudeAlignmentTypeEnum.AgM9391AmplitudeAlignmentTypeCurrentState);
                doVsaAlign[dataLog.testIndex] = false;
            }

            // Wait for VSG to complete
            vsg.Wait();

            dataLog.freqString = String.Format("{0:F3}", centerFreq / 1e6);

            // Log Results
            string[] dataNames = { "setTestFreq Time" };
            string[] dataUnits = { "ms" };
            string[] numFormats = { "0" };
            double[] dataValues = { sw1.ElapsedMilliseconds };
            dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
        }

        //
        // setXappFreq
        // Sets the Frequency of the M9300 X-Series software
        public void setXappFreq(double centerFreq)
        {
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            // Set VSA to frequency changing the RF frequency
            string cmd = string.Format(":FREQ:CENT {0} Hz\n", centerFreq);
            AgVisa32.viPrintf(xApp, cmd);

            // Log Results
            string[] dataNames = { "setXappFreq Time" };
            string[] dataUnits = { "ms" };
            string[] numFormats = { "0" };
            double[] dataValues = { sw1.ElapsedMilliseconds };
            dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
        }

        //
        // setTestFreqWithBaseband
        // Sets the Frequency of the VSG using the baseband offset frequency property of the M9381A
        // Sets the Frequency of the VSA using the RF frequency (Offset frequency will be used for ACPR offsets)
        public void setTestFreqWithBaseband(double centerFreq, double offsetFreq)
        {
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            // Set VSA to frequency changing the RF frequency
            Vsa.RF.Frequency = centerFreq + offsetFreq;
            Vsa.PowerAcquisition.OffsetFrequency = 0;
            Vsa.Apply();
            // Set source to Frequency using baseband frequency offset
            Vsg.Modulation.BasebandFrequency = offsetFreq;
            Vsg.Apply();
            Vsa.WaitUntilSettled(10);
            dataLog.freqString = String.Format("{0:F3}", (centerFreq + offsetFreq) / 1e6);

            // Log Results
            string[] dataNames = { "setFreqWithBaseband Time" };
            string[] dataUnits = { "ms" };
            string[] numFormats = { "0" };
            double[] dataValues = { sw1.ElapsedMilliseconds };
            dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
        }

        public void stopModulation()
        {
            Vsg.Modulation.Stop();
        }

        public void runDcOffsetCal(double[] freqs, string waveform)
        {
                Vsg.Modulation.PlayArb(waveform, AgM938xStartEventEnum.AgM938xStartEventImmediate);
                // Loop over Frequencies
                for (int i = 0; i < freqs.Length; i++)
                {
                    // Set VSG and VSA to test frequency
                    Vsg.RF.Frequency = freqs[i];
                    Vsg.Modulation.BasebandFrequency = 0;
                    Vsg.Apply();
                    Vsg.Calibration.IQAlignment.AlignIQAtDC();
                }
        }

        public void setVsgIqDelay(double delay, bool apply = false)
        {
            Vsg.Modules2.Modulator2.StartPlaybackOn10MHzClock = true;
            Vsg.Modulation2.BasebandDelay = delay;
            if (apply)
                Vsg.Apply();
        }

        public void setVsgIqScale(double scale, bool apply = false)
        {
            Vsg.Modulation.Scale = scale;
            if (apply)
                Vsg.Apply();
        }

        //
        // servoInputPower
        // Adjust the VSG source level using the baseband level property of M9381A
        // If servo can't converge servo Count will be set to -1
        public int servoInputPower(double targetPout, double poutMargin)
        {
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            double measPout = -200;
            bool overload = false;
            dataLog.servoCount[dataLog.testIndex] = 0;
            // Servo Loop
            do
            {
                // Measure before adjuString source in case last power level is correct
                Vsa.Arm();
                Vsa.PowerAcquisition.ReadPower(0, AgM9391PowerUnitsEnum.AgM9391PowerUnitsdBm, ref measPout, ref overload);
                if (overload) throw (new Exception("Overload at servoInputPower"));
                if (measPout < Vsa.RF.Power - 40)
                {
                    Delay(1);
                    Vsa.Apply();
                    Vsa.Arm();
                    double tempPwr = -200;
                    Vsa.PowerAcquisition.ReadPower(0, AgM9391PowerUnitsEnum.AgM9391PowerUnitsdBm, ref tempPwr, ref overload);
                    if (overload) throw (new Exception("Overload at servoInputPower"));
                    measPout = tempPwr;
                }
                measPout += getOutputAtten(Vsa.RF.Frequency + Vsa.PowerAcquisition.OffsetFrequency, 0);
                if (Math.Abs(measPout - targetPout) > poutMargin)
                {
                    double basebandPower = -10;
                    if (!simulateHardware) basebandPower = Vsg.Modulation.BasebandPower + targetPout - measPout;
                    Vsg.Modulation.BasebandPower = Math.Min(basebandPower, 0.0);
                    if (alignVsg == true)
                    {
                        Vsg.Calibration.PowerSearch.UsePowerSearchResult(rfOffset[dataLog.testIndex], scaleOffset[dataLog.testIndex]);
                    }
                    Vsg.Apply();
                }
                dataLog.servoCount[dataLog.testIndex]++;
            } while (Math.Abs(measPout - targetPout) > poutMargin && Vsg.Modulation.BasebandPower < 0 && simulateHardware == false);
            // Set Servo count to -1 if loop fails to converge
            if (Vsg.Modulation.BasebandPower == 0) dataLog.servoCount[dataLog.testIndex] = -1;

            // Set measurement results
            dataLog.chanPower[dataLog.testIndex] = measPout;
            dataLog.chanPowerOverload[dataLog.testIndex] = overload;
            dataLog.inputPower[dataLog.testIndex] = Vsg.RF.Level + Vsg.Modulation.BasebandPower -
                getInputAtten(Vsg.RF.Frequency + Vsg.Modulation.BasebandFrequency, 0);
            dataLog.gain[dataLog.testIndex] = dataLog.chanPower[dataLog.testIndex] - dataLog.inputPower[dataLog.testIndex];

            // Log Results
            string[] dataNames = { "Pin", "Pout", "Gain", "Count", "Servo Time" };
            string[] dataUnits = { "dBm", "dBm", "dB", "Iterations", "ms" };
            string[] numFormats = { "0.00", "0.00", "0.00", "0", "0" };
            double[] dataValues = { dataLog.inputPower[dataLog.testIndex], dataLog.chanPower[dataLog.testIndex], 
                                      dataLog.gain[dataLog.testIndex], dataLog.servoCount[dataLog.testIndex],
                                      sw1.ElapsedMilliseconds };
            dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);

            return 0;
        }


        //
        // servoInputPowerFft
        // Adjust the VSG source level using the baseband level property of M9381A
        // If servo can't converge servo Count will be set to -1
        // Measurements are made using the VSA IQ mode and real time FFT
        public int servoInputPowerFft(double targetPout, double poutMargin)
        {
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            int spectrumCount = (int)(pwrDuration * acprSpan * 1.25 / (int)fftSize);// seconds * samples/second / fftSize
            Vsa.FFTAcquisition.Configure(fftSize, acprSpan * 1.25, spectrumCount, fftWindow, 0);
            Vsa.Apply();
            bool overload = false;
            double measPout = 0;
            dataLog.servoCount[dataLog.testIndex] = 0;
            // Servo Loop
            do
            {
                // Measure before adjusting source in case last power level is correct
                if (dataLog.servoCount[dataLog.testIndex] == 2 && acprDuration > pwrDuration)
                {
                    spectrumCount = (int)(acprDuration * acprSpan * 1.25 / (int)fftSize);// seconds * samples/second / fftSize
                    Vsa.FFTAcquisition.Configure(fftSize, acprSpan * 1.25, spectrumCount, fftWindow, 0);
                    Vsa.Apply();
                }
                measPout = 0;
                Vsa.Arm();
                Vsa.FFTAcquisition.ReadMagnitudeData(0, ref fftData, ref overload);

                if (overload) throw (new Exception("Overload at servoInputPowerIq"));
                double fCenter = Vsa.RF.Frequency;
                fDelta = acprSpan * 1.25 / (int)fftSize;
                fStart = fCenter - ((int)fftSize / 2 * fDelta);

                bool isRrcFilter = false;
                bool isGausFilter = false;
                double channelBw = FilterBw;
                if (FilterType == AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeRootRaisedCosine)
                {
                    isRrcFilter = true;
                    // use a wider BW for RRC filter to get the power in the filter skirt
                    channelBw = FilterBw * 1.3;
                }
                else if (FilterType == AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeGaussian)
                {
                    isGausFilter = true;
                    // use a wider BW for Gaussian filter to get the power in the filter skirt
                    channelBw = FilterBw * 2.0;
                }

                // calculate start and stop bin for Reference power, including any fractional bin
                double fBinStart = fCenter - channelBw / 2;
                int startBin = (int)Math.Floor((fBinStart - fStart) / fDelta);
                int stopBin = (int)Math.Ceiling((fBinStart - fStart + channelBw) / fDelta);
                for (int j = startBin; j <= stopBin; j++)
                {
                    if (isRrcFilter)
                    {
                        double binFreq = (fStart + j * fDelta) - fCenter;
                        double binScale = getRrcFilterWeightSquared(binFreq, FilterBw, FilterAlpha);
                        measPout += fftData[j] * binScale;
                    }
                    else if (isGausFilter)
                    {
                        double binFreq = (fStart + j * fDelta) - fCenter;
                        double binScale = getGausFilterWeight(binFreq, FilterBw);
                        measPout += fftData[j] * binScale;
                    }
                    else
                    {
                        // for RECT filtered channels, scale by percentage of bin inside channel for first and last bin
                        if (j == startBin)
                        {
                            double binScale = ((fBinStart - fStart) / fDelta - startBin);
                            measPout += fftData[j] * binScale;
                        }
                        else if (j == stopBin)
                        {
                            double binScale = (stopBin - (fBinStart - fStart + channelBw) / fDelta);
                            measPout += fftData[j] * binScale;
                        }
                        else
                            measPout += fftData[j];
                    }

                }
                measPout = 10 * Math.Log10(measPout);
                measPout += getOutputAtten(Vsa.RF.Frequency + Vsa.PowerAcquisition.OffsetFrequency, 0);
                // Correct for the ENBW of FFT Window choice
                measPout -= 10 * Math.Log10(windowEnbw);
                if (Math.Abs(measPout - targetPout) > poutMargin)
                {
                    double basebandPower = -10;
                    if (!simulateHardware) basebandPower = Vsg.Modulation.BasebandPower + targetPout - measPout;
                    Vsg.Modulation.BasebandPower = Math.Min(basebandPower, 0.0);
                    if (alignVsg == true)
                    {
                        Vsg.Calibration.PowerSearch.UsePowerSearchResult(rfOffset[dataLog.testIndex], scaleOffset[dataLog.testIndex]);
                    }
                    Vsg.Apply();
                }
                dataLog.servoCount[dataLog.testIndex]++;
            } while (Math.Abs(measPout - targetPout) > poutMargin && Vsg.Modulation.BasebandPower < 0 && simulateHardware == false);

            // If the servo loop converged in 2 steps or less, make another measurement with the longer duration for ACPR calculation
            // Don't, if the ACPR and Power durations are the same
            if (dataLog.servoCount[dataLog.testIndex] < 3 && acprDuration > pwrDuration)
            {
                spectrumCount = (int)(acprDuration * acprSpan * 1.25 / (int)fftSize);// seconds * samples/second / fftSize
                Vsa.FFTAcquisition.Configure(fftSize, acprSpan * 1.25, spectrumCount, fftWindow, 0);
                Vsa.Apply();
                Vsa.Arm();
                Vsa.FFTAcquisition.ReadMagnitudeData(0, ref fftData, ref overload);
                if (overload) throw (new Exception("Overload at servoInputPower"));
            }

            // Set Servo count to -1 if loop fails to converge
            if (Vsg.Modulation.BasebandPower == 0) dataLog.servoCount[dataLog.testIndex] = -1;

            // Set measurement results
            dataLog.chanPower[dataLog.testIndex] = measPout;
            dataLog.chanPowerOverload[dataLog.testIndex] = overload;
            dataLog.inputPower[dataLog.testIndex] = Vsg.RF.Level + Vsg.Modulation.BasebandPower -
                getInputAtten(Vsg.RF.Frequency + Vsg.Modulation.BasebandFrequency, 0);
            dataLog.gain[dataLog.testIndex] = dataLog.chanPower[dataLog.testIndex] - dataLog.inputPower[dataLog.testIndex];

            // Log Results
            string[] dataNames = { "Pin", "Pout", "Gain", "Count", "Servo Time" };
            string[] dataUnits = { "dBm", "dBm", "dB", "Iterations", "ms" };
            string[] numFormats = { "0.00", "0.00", "0.00", "0", "0" };
            double[] dataValues = { dataLog.inputPower[dataLog.testIndex], dataLog.chanPower[dataLog.testIndex], 
                                      dataLog.gain[dataLog.testIndex], dataLog.servoCount[dataLog.testIndex], 
                                      sw1.ElapsedMilliseconds };
            dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);

            return 0;
        }

        // lockM9391
        // Locks the M9391A to be used directly by the IVI driver, preventing any measurements to be run by the XApp software
        public void lockVsa()
        {
            if (!useM9300) return;
            // Stop the X-Apps and make sure they are done with the hardware
            AgVisa32.viPrintf(xApp, ":SYSTem:KLOCk 1;\n");
            AgVisa32.viPrintf(xApp, ":ABORT;*OPC?\n");
            string s;
            AgVisa32.viRead(xApp, out s, 800);
            // Lock the resource back to the driver
            M9000.Checkout(_key, out m9000Resource);
            Vsa.RestoreDefaultProperties();
        }

        // unlockM9391
        // Unlocks the M9391A from the IVI driver, allowing the X-App software to make measurements
        public void unlockVsa()
        {
            if (!useM9300) return;
            // Unlock the resource so xApp can use it
            if (m9000Resource != null) M9000.Checkin(m9000Resource);
        }

        public void sendXappScpi(string command)
        {
            AgVisa32.viPrintf(xApp, command);
        }

        public string getXAppScpi(int length)
        {
            string s = "";
            AgVisa32.viRead(xApp, out s, length);
            return s;
        }

        #endregion

        #region Common Measurement Routines

        public int measPoutFft()
        {
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            bool overload = false;
            double measPout = 0;
            Vsa.Arm();
            Vsa.FFTAcquisition.ReadMagnitudeData(0, ref fftData, ref overload);

            if (overload) throw (new Exception("Overload at servoInputPowerIq"));
            double fCenter = Vsa.RF.Frequency;
            fDelta = acprSpan * 1.25 / (int)fftSize;
            fStart = fCenter - ((int)fftSize / 2 * fDelta);

            bool isRrcFilter = false;
            bool isGausFilter = false;
            double channelBw = FilterBw;
            if (FilterType == AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeRootRaisedCosine)
            {
                isRrcFilter = true;
                // use a wider BW for RRC filter to get the power in the filter skirt
                channelBw = FilterBw * 1.3;
            }
            else if (FilterType == AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeGaussian)
            {
                isGausFilter = true;
                // use a wider BW for Gaussian filter to get the power in the filter skirt
                channelBw = FilterBw * 2.0;
            }

            // calculate start and stop bin for Reference power, including any fractional bin
            double fBinStart = fCenter - channelBw / 2;
            int startBin = (int)Math.Floor((fBinStart - fStart) / fDelta);
            int stopBin = (int)Math.Ceiling((fBinStart - fStart + channelBw) / fDelta);
            for (int j = startBin; j <= stopBin; j++)
            {
                if (isRrcFilter)
                {
                    double binFreq = (fStart + j * fDelta) - fCenter;
                    double binScale = getRrcFilterWeightSquared(binFreq, FilterBw, FilterAlpha);
                    measPout += fftData[j] * binScale;
                }
                else if (isGausFilter)
                {
                    double binFreq = (fStart + j * fDelta) - fCenter;
                    double binScale = getGausFilterWeight(binFreq, FilterBw);
                    measPout += fftData[j] * binScale;
                }
                else
                {
                    // for RECT filtered channels, scale by percentage of bin inside channel for first and last bin
                    if (j == startBin)
                    {
                        double binScale = ((fBinStart - fStart) / fDelta - startBin);
                        measPout += fftData[j] * binScale;
                    }
                    else if (j == stopBin)
                    {
                        double binScale = (stopBin - (fBinStart - fStart + channelBw) / fDelta);
                        measPout += fftData[j] * binScale;
                    }
                    else
                        measPout += fftData[j];
                }

            }
            measPout = 10 * Math.Log10(measPout);
            measPout += getOutputAtten(Vsa.RF.Frequency + Vsa.PowerAcquisition.OffsetFrequency, 0);
            // Correct for the ENBW of FFT Window choice
            measPout -= 10 * Math.Log10(windowEnbw);
            // Set measurement results
            dataLog.chanPower[dataLog.testIndex] = measPout;
            dataLog.chanPowerOverload[dataLog.testIndex] = overload;

            // Log Results

            string[] dataNames = { "Pout", "Pout Time" };
            string[] dataUnits = { "dBm", "ms" };
            string[] numFormats = { "0.00", "0.000" };
            double[] dataValues = { dataLog.chanPower[dataLog.testIndex], (double)sw1.ElapsedTicks / (double)Stopwatch.Frequency * 1000.00 };
            dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);

            return 0;
        }

        public void measStdPout()
        {
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            // Measure Power Output
            Vsa.Arm();
            double measPout = -200;
            bool overload = false;

            Vsa.PowerAcquisition.ReadPower(0, AgM9391PowerUnitsEnum.AgM9391PowerUnitsdBm, ref measPout, ref overload);
            if (overload) throw (new Exception("Overload at measStdPout"));
            measPout += getOutputAtten(Vsa.RF.Frequency + Vsa.PowerAcquisition.OffsetFrequency, 0);
            dataLog.chanPower[dataLog.testIndex] = measPout;

            // Log Results
            string[] dataNames = { "Pout", "Pout Time" };
            string[] dataUnits = { "dBm", "ms" };
            string[] numFormats = { "0.00", "0" };
            double[] dataValues = { dataLog.chanPower[dataLog.testIndex], sw1.ElapsedMilliseconds };
            dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
        }

        public void measStdAcpr()
        {
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            // "Standard" ACPR measurement using a single channel filter
            double measPout = -200;
            bool overload = false;
            // Measure the reference power in the ACPR bandwidth
            for (int j = 0; j < numAcprMeas; j++)
            {
                if (useVsaOffsetFreq)
                    Vsa.PowerAcquisition.OffsetFrequency = acprOffsetFreq[j];
                else
                    Vsa.RF.Frequency = Vsg.RF.Frequency + Vsg.Modulation.BasebandFrequency + acprOffsetFreq[j];

                Vsa.PowerAcquisition.Duration = acprDuration;
                Vsa.Apply();
                Vsa.Arm();
                Vsa.PowerAcquisition.ReadPower(0, AgM9391PowerUnitsEnum.AgM9391PowerUnitsdBm,
                        ref measPout, ref overload);
                if (overload) throw (new Exception("Overload at measStedAcpr"));
                measPout += getOutputAtten(Vsa.RF.Frequency + Vsa.PowerAcquisition.OffsetFrequency, 0);
                dataLog.measAcpr[dataLog.testIndex, j] = measPout - dataLog.chanPower[dataLog.testIndex];
                dataLog.measAcprOverload[dataLog.testIndex, j] = overload;
            }
            Vsa.PowerAcquisition.OffsetFrequency = 0;

            // Log Results
            if (numAcprMeas == 2)
            {
                string[] dataNames = { "ACPR1L", "ACPR1U", "ACPR Time" };
                string[] dataUnits = { "dBc", "dBc", "ms" };
                string[] numFormats = { "0.00", "0.00", "0" };
                double[] dataValues = { dataLog.measAcpr[dataLog.testIndex, 0], dataLog.measAcpr[dataLog.testIndex, 1], 
                                          sw1.ElapsedMilliseconds };
                dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
            }
            else if (numAcprMeas == 4)
            {
                string[] dataNames = { "ACPR1L", "ACPR1U", "ACPR2L", "ACPR2U", "ACPR Time" };
                string[] dataUnits = { "dBc", "dBc", "dBc", "dBc", "ms" };
                string[] numFormats = { "0.00", "0.00", "0.00", "0.00", "0" };
                double[] dataValues = { dataLog.measAcpr[dataLog.testIndex, 0], dataLog.measAcpr[dataLog.testIndex, 1], 
                                      dataLog.measAcpr[dataLog.testIndex, 2], dataLog.measAcpr[dataLog.testIndex, 3],
                                      sw1.ElapsedMilliseconds };
                dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
            }
            else if (numAcprMeas == 6)
            {
                string[] dataNames = { "ACPR1L", "ACPR1U", "ACPR2L", "ACPR2U", "ACPR3L", "ACPR3U", "ACPR Time" };
                string[] dataUnits = { "dBc", "dBc", "dBc", "dBc", "dBc", "dBc", "ms" };
                string[] numFormats = { "0.00", "0.00", "0.00", "0.00", "0.00", "0.00", "0" };
                double[] dataValues = { dataLog.measAcpr[dataLog.testIndex, 0], dataLog.measAcpr[dataLog.testIndex, 1], 
                                      dataLog.measAcpr[dataLog.testIndex, 2], dataLog.measAcpr[dataLog.testIndex, 3], 
                                      dataLog.measAcpr[dataLog.testIndex, 4], dataLog.measAcpr[dataLog.testIndex, 5], 
                                      sw1.ElapsedMilliseconds };
                dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
            }
        }

        public void measFftAcpr()
        {
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            double fCenter = Vsa.RF.Frequency;
            double[] mChannelPowers = new double[numAcprMeas];
            double[] mChanneldBc = new double[numAcprMeas];
            bool isRrcFilter = false;
            double channelBw = FilterBw;
            double fBinStart;
            int startBin;
            int stopBin;

            double measPout = 0;
            double refPower = 0;
            if (!fftServo)
            {
                // Servo was done with Channel Power measurement, need to make one FFT Acquisition
                // Configure the Acquisition to use Real Time FFT Mode
                Vsa.AcquisitionMode = AgM9391AcquisitionModeEnum.AgM9391AcquisitionModeFFT;
                int spectrumCount = (int)(acprDuration * acprSpan * 1.25 / (int)fftSize);// seconds * samples/second / fftSize
                Vsa.FFTAcquisition.Configure(fftSize, acprSpan * 1.25, spectrumCount, fftWindow, 0);

                // Set the Effective Noise Bandwidth based on Window Type
                // 1.0 for Uniform, 1.5 for Hann, 2.215 for GaussTop, 3.8194 for FlatTop 
                switch (fftWindow)
                {
                    case AgM9391FFTWindowShapeEnum.AgM9391FFTWindowShapeUniform:
                        {
                            windowEnbw = 1.0;
                            break;
                        }
                    case AgM9391FFTWindowShapeEnum.AgM9391FFTWindowShapeHann:
                        {
                            windowEnbw = 1.5;
                            break;
                        }
                    case AgM9391FFTWindowShapeEnum.AgM9391FFTWindowShapeGaussian:
                        {
                            windowEnbw = 2.215;
                            break;
                        }
                    case AgM9391FFTWindowShapeEnum.AgM9391FFTWindowShapeFlatTop:
                        {
                            windowEnbw = 3.8194;
                            break;
                        }
                    default:
                        {
                            throw (new Exception("Window Mode Not Supported:" + fftWindow.ToString()));
                        }
                }
                bool overload = false;
                Vsa.Apply();
                Vsa.Arm();
                Vsa.FFTAcquisition.ReadMagnitudeData(0, ref fftData, ref overload);
                // Switch back to power mode
                Vsa.AcquisitionMode = Agilent.AgM9391.Interop.AgM9391AcquisitionModeEnum.AgM9391AcquisitionModePower;
                Vsa.Apply();

                if (overload) throw (new Exception("Overload at measLteFftAcpr"));
                fCenter = Vsa.RF.Frequency;
                fDelta = acprSpan * 1.25 / (int)fftSize;
                fStart = fCenter - ((int)fftSize / 2 * fDelta) - fDelta / 2;

                if (FilterType == AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeRootRaisedCosine)
                {
                    isRrcFilter = true;
                    // use a wider BW for RRC filter to get the power in the filter skirt
                    channelBw = FilterBw * 1.3;
                }

                // calculate start and stop bin for Reference power, including any fractional bin
                fBinStart = fCenter - channelBw / 2;
                startBin = (int)Math.Floor((fBinStart - fStart) / fDelta);
                stopBin = (int)Math.Floor((fBinStart - fStart + channelBw) / fDelta);
                for (int j = startBin; j <= stopBin; j++)
                {
                    if (isRrcFilter)
                    {
                        double binFreq = (fStart + j * fDelta) - fCenter;
                        double binScale = getRrcFilterWeightSquared(binFreq, FilterBw, FilterAlpha);
                        measPout += fftData[j] * binScale;
                    }
                    else
                    {
                        // for RECT filtered channels, scale by percentage of bin inside channel for first and last bin
                        if (j == startBin)
                        {
                            double binScale = ((fBinStart - fStart) / fDelta - startBin);
                            measPout += fftData[j] * binScale;
                        }
                        else if (j == stopBin)
                        {
                            double binScale = (stopBin + 1 - (fBinStart - fStart + channelBw) / fDelta);
                            measPout += fftData[j] * binScale;
                        }
                        else
                            measPout += fftData[j];
                    }

                }
                measPout = 10 * Math.Log10(measPout);
                refPower = measPout;
                // Correct measPout to compare to Channel power
                measPout += getOutputAtten(Vsa.RF.Frequency + Vsa.PowerAcquisition.OffsetFrequency, 0);
                // Correct for the ENBW of FFT Window choice
                measPout -= 10 * Math.Log10(windowEnbw);
            }
            else
            {
                // Use the reference power from the Servo Loop, removing the output Attenuation and window ENBW factors since ACPR is relative
                refPower = dataLog.chanPower[dataLog.testIndex] - getOutputAtten(Vsa.RF.Frequency + Vsa.PowerAcquisition.OffsetFrequency, 0) + 10 * Math.Log10(windowEnbw);
            }

            channelBw = acprFilterBw;
            if (acprFilterType == AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeRootRaisedCosine)
            {
                isRrcFilter = true;
                // use a wider BW for RRC filter to get the power in the filter skirt
                channelBw = acprFilterBw * 1.3;
            }

            // Loop over channel offsets to calculate power in each channel
            for (int i = 0; i < numAcprMeas; i++)
            {
                // calculate start and stop bin for offset, including any fractional bin
                fBinStart = fCenter + acprOffsetFreq[i] - channelBw / 2;
                startBin = (int)Math.Floor((fBinStart - fStart) / fDelta);
                stopBin = (int)Math.Floor((fBinStart - fStart + channelBw) / fDelta);
                for (int j = startBin; j <= stopBin; j++)
                {
                    if (isRrcFilter)
                    {
                        double binFreq = (fStart + j * fDelta) - fCenter - acprOffsetFreq[i];
                        double binScale = getRrcFilterWeightSquared(binFreq, acprFilterBw, acprFilterAlpha);
                        mChannelPowers[i] += fftData[j] * binScale;
                    }
                    else
                    {
                        // for RECT filtered channels, scale by percentage of bin inside channel for first and last bin
                        if (j == startBin)
                        {
                            double binScale = ((fBinStart - fStart) / fDelta - startBin);
                            mChannelPowers[i] += fftData[j] * binScale;
                        }
                        else if (j == stopBin)
                        {
                            double binScale = (stopBin + 1 - (fBinStart - fStart + channelBw) / fDelta);
                            mChannelPowers[i] += fftData[j] * binScale;
                        }
                        else
                            mChannelPowers[i] += fftData[j];
                    }
                }
                mChannelPowers[i] = 10 * Math.Log10(mChannelPowers[i]);
                mChanneldBc[i] = mChannelPowers[i] - refPower;
            }

            // Log Results
            if (numAcprMeas == 2)
            {
                string[] dataNames = { "ACPR1L", "ACPR1U", "ACPR Time" };
                string[] dataUnits = { "dBc", "dBc", "ms" };
                string[] numFormats = { "0.00", "0.00", "0" };
                //double[] dataValues = {dataLog.measAcpr[dataLog.testIndex, 0],dataLog.measAcpr[dataLog.testIndex, 1], sw1.ElapsedMilliseconds };
                double[] dataValues = { mChanneldBc[0], mChanneldBc[1], sw1.ElapsedMilliseconds };
                dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
            }
            else if (numAcprMeas == 4)
            {
                string[] dataNames = { "ACPR1L", "ACPR1U", "ACPR2L", "ACPR2U", "ACPR Time" };
                string[] dataUnits = { "dBc", "dBc", "dBc", "dBc", "ms" };
                string[] numFormats = { "0.00", "0.00", "0.00", "0.00", "0" };
                //double[] dataValues = {dataLog.measAcpr[dataLog.testIndex, 0],dataLog.measAcpr[dataLog.testIndex, 1], 
                //                     dataLog.measAcpr[dataLog.testIndex, 2],dataLog.measAcpr[dataLog.testIndex, 3], sw1.ElapsedMilliseconds };
                double[] dataValues = { mChanneldBc[0], mChanneldBc[1], 
                                      mChanneldBc[2], mChanneldBc[3], sw1.ElapsedMilliseconds };
                dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
            }
            else if (numAcprMeas == 6)
            {
                string[] dataNames = { "ACPR1L", "ACPR1U", "ACPR2L", "ACPR2U", "ACPR3L", "ACPR3U", "ACPR Time" };
                string[] dataUnits = { "dBc", "dBc", "dBc", "dBc", "dBc", "dBc", "ms" };
                string[] numFormats = { "0.00", "0.00", "0.00", "0.00", "0.00", "0.00", "0" };
                //double[] dataValues = {dataLog.measAcpr[dataLog.testIndex, 0],dataLog.measAcpr[dataLog.testIndex, 1], 
                //                     dataLog.measAcpr[dataLog.testIndex, 2],dataLog.measAcpr[dataLog.testIndex, 3], 
                //                     dataLog.measAcpr[dataLog.testIndex, 4],dataLog.measAcpr[dataLog.testIndex, 5], sw1.ElapsedMilliseconds };
                double[] dataValues = { mChanneldBc[0], mChanneldBc[1], 
                                      mChanneldBc[2], mChanneldBc[3], 
                                      mChanneldBc[4], mChanneldBc[5], sw1.ElapsedMilliseconds };
                dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
            }
        }

        public void measSpecHarms(bool meas2, bool meas3)
        {
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            bool overload = false;
            // Measure F2
            // For Harmonic measurements on Fc above 1.35 GHz, we will switch to image protect mode and set the refernce level to +5 dBm.
            // This level will keep the input amp in the M9350A off and the image protect filter will reject the funamental
            // from the input of the first mixer, allowing a more sensitive range to be used for harmonics.
            // For Fc below 1.35 GHz, we will increase the referendce level by 20 dB, or to 30 dBm, which ever is less to eliminate any
            // internally generated harmonics

            double tempRefLevel = Vsa.RF.Power;
            double tempRmsLevel = Vsa.RF.PeakToAverage;
            AgM9391AcquisitionModeEnum previousAcqMode = Vsa.AcquisitionMode;
            if (Vsg.RF.Frequency > 1.35e9)
            {
                Vsa.RF.Power = 5;
                Vsa.RF.PeakToAverage = 0;
                Vsa.RF.Conversion = AgM9391ConversionEnum.AgM9391ConversionImageProtect;
            }
            else
            {
                Vsa.RF.Power = Math.Min(tempRefLevel + 20, 30);
                Vsa.RF.PeakToAverage = 0;
            }
            Vsa.AcquisitionMode = AgM9391AcquisitionModeEnum.AgM9391AcquisitionModeSpectrum;
            Vsa.SpectrumAcquisition.Span = 30e6;
            Vsa.SpectrumAcquisition.ResolutionBandwidth = 1e6;
            Vsa.SpectrumAcquisition.FFTWindowShape = AgM9391FFTWindowShapeEnum.AgM9391FFTWindowShapeHDRFlatTop;
            if (meas2)
            {
                Vsa.SpectrumAcquisition.Averaging.Configure(AgM9391SpectrumAveragingEnum.AgM9391SpectrumAveragingTime, 50, .0005, 0.5);
                Vsa.RF.Frequency = 2 * (Vsg.RF.Frequency + Vsg.Modulation.BasebandFrequency);
                Vsa.Apply();
                Array.Resize(ref spectrumData, Vsa.SpectrumAcquisition.Bins);
                double fStart = 0;
                double fDelta = 0;
                Vsa.Arm();
                Vsa.SpectrumAcquisition.ReadPowerSpectrum(0, AgM9391PowerUnitsEnum.AgM9391PowerUnitsdBm,
                    ref spectrumData, ref overload, ref fStart, ref fDelta);
                dataLog.harm2[dataLog.testIndex] = spectrumData.Max();
                dataLog.harm2[dataLog.testIndex] += getOutputAtten(Vsa.RF.Frequency + Vsa.PowerAcquisition.OffsetFrequency, 0);
            }
            // Measure F3
            if (meas3)
            {
                Vsa.SpectrumAcquisition.Averaging.Configure(AgM9391SpectrumAveragingEnum.AgM9391SpectrumAveragingTime, 200, .001, 0.5);
                Vsa.RF.Frequency = 3 * (Vsg.RF.Frequency + Vsg.Modulation.BasebandFrequency);
                double fStart = 0;
                double fDelta = 0;
                Vsa.Apply();
                Array.Resize(ref spectrumData, Vsa.SpectrumAcquisition.Bins);
                Vsa.Arm();
                Vsa.SpectrumAcquisition.ReadPowerSpectrum(0, AgM9391PowerUnitsEnum.AgM9391PowerUnitsdBm,
                    ref spectrumData, ref overload, ref fStart, ref fDelta);
                dataLog.harm3[dataLog.testIndex] = spectrumData.Max();
                dataLog.harm3[dataLog.testIndex] += getOutputAtten(Vsa.RF.Frequency + Vsa.PowerAcquisition.OffsetFrequency, 0);
            }

            // Switch back to previous acquisition mode, auto conversion mode and normal power settings
            Vsa.AcquisitionMode = previousAcqMode;
            Vsa.RF.Power = tempRefLevel;
            Vsa.RF.PeakToAverage = tempRmsLevel;
            Vsa.RF.Conversion = AgM9391ConversionEnum.AgM9391ConversionAuto;
            Vsa.RF.Frequency = (Vsg.RF.Frequency + Vsg.Modulation.BasebandFrequency);
            Vsa.Apply();

            // Log Results
            if (meas2 && meas3)
            {
                string[] dataNames = { "Second Harmonic", "Third Harmonic", "Harmonics Time" };
                string[] dataUnits = { "dBm", "dBm", "ms" };
                string[] numFormats = { "0.00", "0.00", "0" };
                double[] dataValues = { dataLog.harm2[dataLog.testIndex], dataLog.harm3[dataLog.testIndex], sw1.ElapsedMilliseconds };
                dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
            }
            else if (meas2)
            {
                string[] dataNames = { "Second Harmonic", "Harmonics Time" };
                string[] dataUnits = { "dBm", "ms" };
                string[] numFormats = { "0.00", "0" };
                double[] dataValues = { dataLog.harm2[dataLog.testIndex], sw1.ElapsedMilliseconds };
                dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
            }
            else if (meas3)
            {
                string[] dataNames = { "Third Harmonic", "Harmonics Time" };
                string[] dataUnits = { "dBm", "ms" };
                string[] numFormats = { "0.00", "0" };
                double[] dataValues = { dataLog.harm3[dataLog.testIndex], sw1.ElapsedMilliseconds };
                dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
            }
        }

        public void measSpecHarms2(bool[] harmsToMeasure)
        {
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            bool overload = false;
            int harmsMeasured = 0;
            double[] harmData = new double[harmsToMeasure.Length];

            AgM9391AcquisitionModeEnum previousAcqMode = Vsa.AcquisitionMode;
            Vsa.AcquisitionMode = AgM9391AcquisitionModeEnum.AgM9391AcquisitionModeSpectrum;
            Vsa.SpectrumAcquisition.Span = 140e6;
            Vsa.SpectrumAcquisition.ResolutionBandwidth = 1e6;
            Vsa.SpectrumAcquisition.FFTWindowShape = AgM9391FFTWindowShapeEnum.AgM9391FFTWindowShapeHDRFlatTop;
            Vsa.SpectrumAcquisition.Averaging.Configure(AgM9391SpectrumAveragingEnum.AgM9391SpectrumAveragingTimePeak, 50, .001, 0.5);

            for (int i = 0; i < harmsToMeasure.Length; i++)
            {
                if (harmsToMeasure[i])
                {
                    Vsa.RF.Frequency = (2 + i) * (Vsg.RF.Frequency + Vsg.Modulation.BasebandFrequency);
                    Vsa.Apply();
                    Array.Resize(ref spectrumData, Vsa.SpectrumAcquisition.Bins);
                    double fStart = 0;
                    double fDelta = 0;
                    Vsa.Arm();
                    Vsa.SpectrumAcquisition.ReadPowerSpectrum(0, AgM9391PowerUnitsEnum.AgM9391PowerUnitsdBm,
                        ref spectrumData, ref overload, ref fStart, ref fDelta);
                    harmData[i] = spectrumData.Max();
                    harmData[i] += getOutputAtten(Vsa.RF.Frequency + Vsa.PowerAcquisition.OffsetFrequency, 0);
                    harmsMeasured++;
                }
            }

            // Switch back to previous acquisition mode, auto conversion mode and normal power settings
            Vsa.AcquisitionMode = previousAcqMode;
            Vsa.RF.Frequency = (Vsg.RF.Frequency + Vsg.Modulation.BasebandFrequency);
            Vsa.Apply();

            long localMeasTime = sw1.ElapsedMilliseconds;

            // Log Results
            string[] dataNames = new string[harmsMeasured + 1];
            string[] dataUnits = new string[harmsMeasured + 1];
            string[] numFormats = new string[harmsMeasured + 1];
            double[] dataValues = new double[harmsMeasured + 1];

            int j = 0;
            for (int i = 0; i < harmsToMeasure.Length; i++)
            {
                if (harmsToMeasure[i])
                {
                    dataNames[j] = "Harmonic " + (i + 2).ToString();
                    dataUnits[j] = "dBm";
                    numFormats[j] = "0.00";
                    dataValues[j] = harmData[i];
                    j++;
                }
            }
            dataNames[dataNames.Length - 1] = "Harm Time";
            dataUnits[dataNames.Length - 1] = "ms";
            numFormats[dataNames.Length - 1] = "0";
            dataValues[dataNames.Length - 1] = sw1.ElapsedMilliseconds;
            dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
        }
        public void measPwrHarms2(bool[] harmsToMeasure)
        {
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            bool overload = false;
            double measPout = -200;
            int harmsMeasured = 0;
            double[] harmData = new double[harmsToMeasure.Length];
            Vsa.PowerAcquisition.Duration = .001;

            for (int i = 0; i < harmsToMeasure.Length; i++)
            {
                if (harmsToMeasure[i])
                {
                    Vsa.RF.Frequency = (2 + i) * (Vsg.RF.Frequency + Vsg.Modulation.BasebandFrequency);
                    Vsa.Apply();
                    Vsa.Arm();
                    Vsa.PowerAcquisition.ReadPower(0, AgM9391PowerUnitsEnum.AgM9391PowerUnitsdBm, ref measPout, ref overload);
                    if (overload) throw (new Exception("Overload at measPwrHarms"));
                    harmData[i] = measPout;
                    harmData[i] += getOutputAtten(Vsa.RF.Frequency + Vsa.PowerAcquisition.OffsetFrequency, 0);
                    harmsMeasured++;
                }
            }

            Vsa.RF.Frequency = (Vsg.RF.Frequency + Vsg.Modulation.BasebandFrequency);
            Vsa.PowerAcquisition.Duration = pwrDuration;
            Vsa.Apply();

            long localMeasTime = sw1.ElapsedMilliseconds;

            // Log Results
            string[] dataNames = new string[harmsMeasured + 1];
            string[] dataUnits = new string[harmsMeasured + 1];
            string[] numFormats = new string[harmsMeasured + 1];
            double[] dataValues = new double[harmsMeasured + 1];

            int j = 0;
            for (int i = 0; i < harmsToMeasure.Length; i++)
            {
                if (harmsToMeasure[i])
                {
                    dataNames[j] = "Harmonic " + (i + 2).ToString();
                    dataUnits[j] = "dBm";
                    numFormats[j] = "0.00";
                    dataValues[j] = harmData[i];
                    j++;
                }
            }
            dataNames[dataNames.Length - 1] = "Harm Time";
            dataUnits[dataNames.Length - 1] = "ms";
            numFormats[dataNames.Length - 1] = "0";
            dataValues[dataNames.Length - 1] = sw1.ElapsedMilliseconds;
            dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
        }

        public void measPwrHarms(bool meas2, bool meas3)
        {
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            double measPout = -200;
            bool overload = false;

            if (meas2)
            {
                // Measure F2
                Vsa.RF.Frequency = 2 * (Vsg.RF.Frequency + Vsg.Modulation.BasebandFrequency);
                Vsa.Apply();
                Vsa.Arm();
                Vsa.PowerAcquisition.ReadPower(0, AgM9391PowerUnitsEnum.AgM9391PowerUnitsdBm, ref measPout, ref overload);
                if (overload) throw (new Exception("Overload at measPwrHarms"));
                dataLog.harm2[dataLog.testIndex] = measPout;
                dataLog.harm2[dataLog.testIndex] += getOutputAtten(Vsa.RF.Frequency + Vsa.PowerAcquisition.OffsetFrequency, 0);
            }

            if (meas3)
            {
                // Measure F3
                Vsa.RF.Frequency = 3 * (Vsg.RF.Frequency + Vsg.Modulation.BasebandFrequency);
                Vsa.Apply();
                Vsa.Arm();
                Vsa.PowerAcquisition.ReadPower(0, AgM9391PowerUnitsEnum.AgM9391PowerUnitsdBm, ref measPout, ref overload);
                if (overload) throw (new Exception("Overload at measPwrHarms"));
                dataLog.harm3[dataLog.testIndex] = measPout;
                dataLog.harm3[dataLog.testIndex] += getOutputAtten(Vsa.RF.Frequency + Vsa.PowerAcquisition.OffsetFrequency, 0);
            }

            // Log Results
            if (meas2 && meas3)
            {
                string[] dataNames = { "Second Harmonic", "Third Harmonic", "Harmonics Time" };
                string[] dataUnits = { "dBm", "dBm", "ms" };
                string[] numFormats = { "0.00", "0.00", "0" };
                double[] dataValues = { dataLog.harm2[dataLog.testIndex], dataLog.harm3[dataLog.testIndex], sw1.ElapsedMilliseconds };
                dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
            }
            else if (meas2)
            {
                string[] dataNames = { "Second Harmonic", "Harmonics Time" };
                string[] dataUnits = { "dBm", "ms" };
                string[] numFormats = { "0.00", "0" };
                double[] dataValues = { dataLog.harm2[dataLog.testIndex], sw1.ElapsedMilliseconds };
                dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
            }
            else if (meas3)
            {
                string[] dataNames = { "Third Harmonic", "Harmonics Time" };
                string[] dataUnits = { "dBm", "ms" };
                string[] numFormats = { "0.00", "0" };
                double[] dataValues = { dataLog.harm3[dataLog.testIndex], sw1.ElapsedMilliseconds };
                dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
            }
        }

        public void measGainComp(double duration, double minPower)
        {
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            // Switch to IQ Acquisition Mode
            Vsa.AcquisitionMode = AgM9391AcquisitionModeEnum.AgM9391AcquisitionModeIQ;
            Vsa.IQAcquisition.ChannelFilter.Configure(FilterType, FilterAlpha, FilterBw);
            int numSamples = (int)(duration * arbSampleRate);
            Vsa.IQAcquisition.Configure(AgM9391SampleSizeEnum.AgM9391SampleSize64Bits, arbSampleRate, numSamples);
            Vsa.Apply();

            // Size Arrays
            Array.Resize(ref iqData, numSamples * 2);
            Array.Resize(ref inPower, numSamples);
            Array.Resize(ref outPower, numSamples);
            Array.Resize(ref gainArray, numSamples);

            // Collect IQ Trace
            Vsa.Arm();
            bool overload = false;
            Vsa.IQAcquisition.ReadIQData(0, AgM9391IQUnitsEnum.AgM9391IQUnitsSquareRootMilliWatts, 0, numSamples, ref iqData, ref overload);
            if (overload) throw (new Exception("Overload at measGainComp"));

            // Calculate Power and Gain Arrays
            double inPwrStep = -1 * minPower / numSamples;
            double maxInPwr = Vsg.RF.Level + Vsg.Modulation.BasebandPower -
                getInputAtten(Vsg.RF.Frequency + Vsg.Modulation.BasebandFrequency, 0);
            for (int i = 0; i < numSamples; i++)
            {
                outPower[i] = 10 * Math.Log10(iqData[2 * i] * iqData[2 * i] + iqData[(2 * i) + 1] * iqData[(2 * i) + 1])
                    + getOutputAtten(Vsa.RF.Frequency, 0);
                inPower[i] = maxInPwr + minPower + i * inPwrStep;
                gainArray[i] = outPower[i] - inPower[i];
            }
            double gainComp1dB = -999;
            bool gainComp1dBfound = false;
            double inPower1dB = -999;
            double gainComp2dB = -999;
            bool gainComp2dBfound = false;
            double inPower2dB = -999;
            double maxGainComp = -999;
            for (int i = numSamples - 1; i > 0; i--)
            {
                double gainComp = gainArray[0] - gainArray[i];
                if (gainComp > 1.0 && !gainComp1dBfound)
                {
                    gainComp1dB = outPower[i];
                    inPower1dB = inPower[i];
                    gainComp1dBfound = true;
                }
                if (gainComp > 2.0 && !gainComp2dBfound)
                {
                    gainComp2dB = outPower[i];
                    inPower2dB = inPower[i];
                    gainComp2dBfound = true;
                }
                if (gainComp > maxGainComp)
                    maxGainComp = gainComp;
            }
            // Switch back to power acquisition mode, auto conversion mode and normal power settings
            Vsa.AcquisitionMode = AgM9391AcquisitionModeEnum.AgM9391AcquisitionModePower;

            // Log Results
            string[] dataNames = { "SS Gain", "1 dB Compression", "1 dB Comp Input", "2 dB Compression", "2 dB Comp Input", "Max Compression", "Gain Comp Time" };
            string[] dataUnits = { "dB", "dBm", "dBm", "dBm", "dBm", "dB", "ms" };
            string[] numFormats = { "0.00", "0.00", "0.00", "0.00", "0.00", "0.00", "0" };
            double[] dataValues = { gainArray[0], gainComp1dB, inPower1dB, gainComp2dB, inPower2dB, maxGainComp, sw1.ElapsedMilliseconds };
            dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
        }

        public void iqDataTo89600File(double duration, string fileName)
        {
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            // Switch to IQ Acquisition Mode
            Vsa.AcquisitionMode = AgM9391AcquisitionModeEnum.AgM9391AcquisitionModeIQ;
            Vsa.IQAcquisition.ChannelFilter.Configure(FilterType, FilterAlpha, FilterBw);
            int numSamples = (int)(duration * arbSampleRate);
            Vsa.IQAcquisition.Configure(AgM9391SampleSizeEnum.AgM9391SampleSize64Bits, arbSampleRate, numSamples);
            Vsa.Apply();

            // Size Arrays
            Array.Resize(ref iqData, numSamples * 2);
            Array.Resize(ref inPower, numSamples);
            Array.Resize(ref outPower, numSamples);
            Array.Resize(ref gainArray, numSamples);

            // Collect IQ Trace
            Vsa.Arm();
            bool overload = false;
            Vsa.IQAcquisition.ReadIQData(0, AgM9391IQUnitsEnum.AgM9391IQUnitsVolts, 0, numSamples, ref iqData, ref overload);
            if (overload) throw (new Exception("Overload at iqDataTo89600File"));

            // Write Data to file
            dataLog.Write89600file("c:\\Temp\\", fileName, iqData, Vsa.RF.Frequency, arbSampleRate);
            // Switch back to power acquisition mode, auto conversion mode and normal power settings
            Vsa.AcquisitionMode = AgM9391AcquisitionModeEnum.AgM9391AcquisitionModePower;

            // Log Results
            string[] dataNames = { "Iq Data to File Time" };
            string[] dataUnits = { "ms" };
            string[] numFormats = { "0" };
            double[] dataValues = { sw1.ElapsedMilliseconds };
            dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
        }

        #endregion

        #region GSM Measurement Routines

        public void measGsmAcpr(int numAverages)
        {
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            double measPout = -200;
            bool overload = false;
            // Measure the reference power in the ACPR bandwidth
            double refPower = -200;
            Vsa.PowerAcquisition.ChannelFilter.Configure(FilterType, FilterAlpha, acprFilterBw);
            Vsa.PowerAcquisition.Duration = acprDuration;
            //M9391.Modules.Synthesizer.PLLMode = AgM9391SynthesizerPLLModeEnum.AgM9391SynthesizerPLLModeBestWideOffset;
            Vsa.Apply();
            // Set the following to true to use the 270 kHz channel power measurement for the ACPR reference
            // If set to false, program will measure refrence power in 30 kHz BW

            if (useChanPwrForRef)
            {
                refPower = dataLog.chanPower[dataLog.testIndex];
            }
            else
            {
                Vsa.Arm();
                Vsa.PowerAcquisition.ReadPower(0, AgM9391PowerUnitsEnum.AgM9391PowerUnitsdBm, ref refPower, ref overload);
                if (overload) throw (new Exception("Overload at measGsmAcpr"));
                refPower += getOutputAtten(Vsa.RF.Frequency + Vsa.PowerAcquisition.OffsetFrequency, 0);
            }

            for (int j = 0; j < numAcprMeas; j++)
            {
                if (useVsaOffsetFreq)
                    Vsa.PowerAcquisition.OffsetFrequency = acprOffsetFreq[j];
                else
                    Vsa.RF.Frequency = Vsg.RF.Frequency + Vsg.Modulation.BasebandFrequency + acprOffsetFreq[j];
                Vsa.Apply();
                double sumPower = 0;
                for (int k = 0; k < numAverages; k++)
                {
                    // Read power in volts squared to allow for averages
                    Vsa.Arm();
                    Vsa.PowerAcquisition.ReadPower(0, AgM9391PowerUnitsEnum.AgM9391PowerUnitsVoltsSquared,
                            ref measPout, ref overload);
                    if (overload) throw (new Exception("Overload at measGsmAcpr"));
                    sumPower += measPout;
                }
                measPout = voltsSquare2dBm(sumPower / numAverages, 50.0);
                measPout += getOutputAtten(Vsa.RF.Frequency + Vsa.PowerAcquisition.OffsetFrequency, 0);
                dataLog.measAcpr[dataLog.testIndex, j] = measPout - refPower;
                dataLog.measAcprOverload[dataLog.testIndex, j] = overload;
            }
            //M9391.Modules.Synthesizer.PLLMode = AgM9391SynthesizerPLLModeEnum.AgM9391SynthesizerPLLModeNormal;
            if (useVsaOffsetFreq)
                Vsa.PowerAcquisition.OffsetFrequency = 0;
            else
                Vsa.RF.Frequency = Vsg.RF.Frequency + Vsg.Modulation.BasebandFrequency + 0;
            Vsa.PowerAcquisition.ChannelFilter.Configure(FilterType, FilterAlpha, FilterBw);
            Vsa.PowerAcquisition.Duration = pwrDuration;

            // Log Results
            string[] dataNames = { "ACPR1L", "ACPR1U", "ACPR2L", "ACPR2U", "ACPR Time" };
            string[] dataUnits = { "dBc", "dBc", "dBc", "dBc", "ms" };
            string[] numFormats = { "0.00", "0.00", "0.00", "0.00", "0" };
            double[] dataValues = { dataLog.measAcpr[dataLog.testIndex, 0], dataLog.measAcpr[dataLog.testIndex, 1], 
                                      dataLog.measAcpr[dataLog.testIndex, 2], dataLog.measAcpr[dataLog.testIndex, 3], 
                                      sw1.ElapsedMilliseconds };
            dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
        }

        public void measGsmFftAcpr(int numAverages)
        {
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            double fCenter = Vsa.RF.Frequency;
            double[] mChannelPowers = new double[numAcprMeas];
            double[] mChanneldBc = new double[numAcprMeas];
            bool isGausFilter = true;
            double channelBw = FilterBw;
            double fBinStart;
            int startBin;
            int stopBin;

            double measPout = 0;
            double refPower = 0;

            ////Allow different settings for GSM/EDGE
            //AgM9391FFTAcquisitionLengthEnum gsmFftSize = fftSize;
            //AgM9391FFTWindowShapeEnum gsmFftWindow = fftWindow;

            if (!fftServo)
            {
                // Servo was done with Channel Power measurement, need to make one FFT Acquisition
                // Configure the Acquisition to use Real Time FFT Mode

                Vsa.AcquisitionMode = AgM9391AcquisitionModeEnum.AgM9391AcquisitionModeFFT;
                int spectrumCount = (int)(acprDuration * acprSpan * 1.25 / (int)fftSize);// seconds * samples/second / fftSize
                Vsa.FFTAcquisition.Configure(fftSize, acprSpan * 1.25, spectrumCount, fftWindow, 0);

                // Set the Effective Noise Bandwidth based on Window Type
                // 1.0 for Uniform, 1.5 for Hann, 2.215 for GaussTop, 3.8194 for FlatTop 
                switch (fftWindow)
                {
                    case AgM9391FFTWindowShapeEnum.AgM9391FFTWindowShapeUniform:
                        {
                            windowEnbw = 1.0;
                            break;
                        }
                    case AgM9391FFTWindowShapeEnum.AgM9391FFTWindowShapeHann:
                        {
                            windowEnbw = 1.5;
                            break;
                        }
                    case AgM9391FFTWindowShapeEnum.AgM9391FFTWindowShapeGaussian:
                        {
                            windowEnbw = 2.215;
                            break;
                        }
                    case AgM9391FFTWindowShapeEnum.AgM9391FFTWindowShapeFlatTop:
                        {
                            windowEnbw = 3.8194;
                            break;
                        }
                    default:
                        {
                            throw (new Exception("Window Mode Not Supported:" + fftWindow.ToString()));
                        }
                }
                Vsa.Apply();
            }

            fCenter = Vsa.RF.Frequency;
            fDelta = acprSpan * 1.25 / (int)fftSize;
            fStart = fCenter - ((int)fftSize / 2 * fDelta) - fDelta / 2;

            for (int k = 0; k < numAverages; k++)
            {
                bool overload = false;

                if (k > 0 || !fftServo)
                {
                    // If the servo was done with FFT, we can skip measuring on the first average
                    // and use the fft data from the servo
                    Vsa.Arm();
                    Vsa.FFTAcquisition.ReadMagnitudeData(0, ref fftData, ref overload);
                }

                if (overload) throw (new Exception("Overload at measGsmFftAcpr"));

                if (FilterType == AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeGaussian)
                {
                    isGausFilter = true;
                    // use a wider BW for RRC filter to get the power in the filter skirt
                    channelBw = FilterBw * 2.0;
                }

                // calculate start and stop bin for Reference power, including any fractional bin
                fBinStart = fCenter - channelBw / 2;
                startBin = (int)Math.Floor((fBinStart - fStart) / fDelta);
                stopBin = (int)Math.Floor((fBinStart - fStart + channelBw) / fDelta);
                for (int j = startBin; j <= stopBin; j++)
                {
                    if (isGausFilter)
                    {
                        double binFreq = (fStart + j * fDelta) - fCenter;
                        double binScale = getGausFilterWeight(binFreq, FilterBw);
                        measPout += fftData[j] * binScale;
                    }
                    else
                    {
                        // for RECT filtered channels, scale by percentage of bin inside channel for first and last bin
                        if (j == startBin)
                        {
                            double binScale = ((fBinStart - fStart) / fDelta - startBin);
                            measPout += fftData[j] * binScale;
                        }
                        else if (j == stopBin)
                        {
                            double binScale = (stopBin + 1 - (fBinStart - fStart + channelBw) / fDelta);
                            measPout += fftData[j] * binScale;
                        }
                        else
                            measPout += fftData[j];
                    }

                }

                channelBw = acprFilterBw;
                if (acprFilterType == AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeGaussian)
                {
                    isGausFilter = true;
                    // use a wider BW for RRC filter to get the power in the filter skirt
                    channelBw = acprFilterBw * 2.0;
                }

                // Loop over channel offsets to calculate power in each channel
                for (int i = 0; i < numAcprMeas; i++)
                {
                    // calculate start and stop bin for offset, including any fractional bin
                    fBinStart = fCenter + acprOffsetFreq[i] - channelBw / 2;
                    startBin = (int)Math.Floor((fBinStart - fStart) / fDelta);
                    stopBin = (int)Math.Floor((fBinStart - fStart + channelBw) / fDelta);
                    for (int j = startBin; j <= stopBin; j++)
                    {
                        if (isGausFilter)
                        {
                            double binFreq = (fStart + j * fDelta) - fCenter - acprOffsetFreq[i];
                            double binScale = getGausFilterWeight(binFreq, FilterBw);
                            mChannelPowers[i] += fftData[j] * binScale;
                        }
                        else
                        {
                            // for RECT filtered channels, scale by percentage of bin inside channel for first and last bin
                            if (j == startBin)
                            {
                                double binScale = ((fBinStart - fStart) / fDelta - startBin);
                                mChannelPowers[i] += fftData[j] * binScale;
                            }
                            else if (j == stopBin)
                            {
                                double binScale = (stopBin + 1 - (fBinStart - fStart + channelBw) / fDelta);
                                mChannelPowers[i] += fftData[j] * binScale;
                            }
                            else
                                mChannelPowers[i] += fftData[j];
                        }
                    }
                }
            }

            measPout = 10 * Math.Log10(measPout / numAverages);
            refPower = measPout;
            // Correct measPout to compare to Channel power
            measPout += getOutputAtten(Vsa.RF.Frequency + Vsa.PowerAcquisition.OffsetFrequency, 0);
            // Correct for the ENBW of FFT Window choice
            measPout -= 10 * Math.Log10(windowEnbw);

            for (int i = 0; i < numAcprMeas; i++)
            {
                mChannelPowers[i] = 10 * Math.Log10(mChannelPowers[i] / numAverages);
                mChanneldBc[i] = mChannelPowers[i] - refPower;
            }

            if (!fftServo)
            {
                // Switch back to power mode
                Vsa.AcquisitionMode = Agilent.AgM9391.Interop.AgM9391AcquisitionModeEnum.AgM9391AcquisitionModePower;
                Vsa.Apply();
            }
            // Log Results
            if (numAcprMeas == 2)
            {
                string[] dataNames = { "ACPR1L", "ACPR1U", "ACPR Time" };
                string[] dataUnits = { "dBc", "dBc", "ms" };
                string[] numFormats = { "0.00", "0.00", "0" };
                //double[] dataValues = {dataLog.measAcpr[dataLog.testIndex, 0],dataLog.measAcpr[dataLog.testIndex, 1], sw1.ElapsedMilliseconds };
                double[] dataValues = { mChanneldBc[0], mChanneldBc[1], sw1.ElapsedMilliseconds };
                dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
            }
            else if (numAcprMeas == 4)
            {
                string[] dataNames = { "ACPR1L", "ACPR1U", "ACPR2L", "ACPR2U", "ACPR Time" };
                string[] dataUnits = { "dBc", "dBc", "dBc", "dBc", "ms" };
                string[] numFormats = { "0.00", "0.00", "0.00", "0.00", "0" };
                //double[] dataValues = {dataLog.measAcpr[dataLog.testIndex, 0],dataLog.measAcpr[dataLog.testIndex, 1], 
                //                     dataLog.measAcpr[dataLog.testIndex, 2],dataLog.measAcpr[dataLog.testIndex, 3], sw1.ElapsedMilliseconds };
                double[] dataValues = { mChanneldBc[0], mChanneldBc[1], 
                                      mChanneldBc[2], mChanneldBc[3], sw1.ElapsedMilliseconds };
                dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
            }
            else if (numAcprMeas == 6)
            {
                string[] dataNames = { "ACPR1L", "ACPR1U", "ACPR2L", "ACPR2U", "ACPR3L", "ACPR3U", "ACPR Time" };
                string[] dataUnits = { "dBc", "dBc", "dBc", "dBc", "dBc", "dBc", "ms" };
                string[] numFormats = { "0.00", "0.00", "0.00", "0.00", "0.00", "0.00", "0" };
                //double[] dataValues = {dataLog.measAcpr[dataLog.testIndex, 0],dataLog.measAcpr[dataLog.testIndex, 1], 
                //                     dataLog.measAcpr[dataLog.testIndex, 2],dataLog.measAcpr[dataLog.testIndex, 3], 
                //                     dataLog.measAcpr[dataLog.testIndex, 4],dataLog.measAcpr[dataLog.testIndex, 5], sw1.ElapsedMilliseconds };
                double[] dataValues = { mChanneldBc[0], mChanneldBc[1], 
                                      mChanneldBc[2], mChanneldBc[3], 
                                      mChanneldBc[4], mChanneldBc[5], sw1.ElapsedMilliseconds };
                dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
            }
        }

        public void configGsmOrfsMeas(string standard, double frequency, double RefLevel, int numAverages, string type)
        {
            if (!useM9300) return;
            AgVisa32.viPrintf(xApp, "INST:SEL GSM\n");
            string cmd = string.Format(":FREQ:CENT {0} Hz\n", frequency);
            AgVisa32.viPrintf(xApp, cmd);
            cmd = string.Format(":POW:RANGE {0} dBm\n", RefLevel + arbRmsValue - getOutputAtten(frequency, 0));
            AgVisa32.viPrintf(xApp, cmd);
            //AgVisa32.viPrintf(xApp, "SENSe:RADio:STANdard:SELect W3GPp\n");
            cmd = string.Format(":RAD:DEV {0}\n", standard); // BTS or MS
            AgVisa32.viPrintf(xApp, cmd);
            // Setup commands from MCD Examples

            cmd = string.Format(":ORFS:AVER:COUN {0}\n", numAverages);
            AgVisa32.viPrintf(xApp, cmd);
            if (numAverages > 1)
                AgVisa32.viPrintf(xApp, ":ORFS:AVER:STATE ON\n");
            else
                AgVisa32.viPrintf(xApp, ":ORFS:AVER:STATE OFF\n");
            cmd = string.Format(":ORFS:TYPE {0}\n", type);
            AgVisa32.viPrintf(xApp, cmd);
            if (vsaTriggerSource == AgM9391TriggerEnum.AgM9391TriggerFrontPanelTrigger1)
            {
                AgVisa32.viPrintf(xApp, ":ORFS:TRIG:SOUR EXT1\n");
            }
            else
            {
                AgVisa32.viPrintf(xApp, ":ORFS:TRIG:SOUR EXT2\n");
            }

            AgVisa32.viPrintf(xApp, ":ORFS:LIST:MOD:STAT 0,0,0,0,1,1\n");
            AgVisa32.viPrintf(xApp, ":ORFS:LIST:SWIT:STAT 1,1,1\n");
            AgVisa32.viPrintf(xApp, ":ORFS:LIST:SEL CUST\n");
            AgVisa32.viPrintf(xApp, "*WAI\n");
        }

        public void measGsmOrfs(int numOffsets, string orfsType)
        {
            if (!useM9300) return;
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            string s = "";
            AgVisa32.viPrintf(xApp, "INIT:ORFS\n");
            AgVisa32.viPrintf(xApp, "*OPC\n");
            AgVisa32.viPrintf(xApp, "FETC:ORFS?\n");
            // Read results as a string and convert
            AgVisa32.viRead(xApp, out s, 8000);
            if (s != String.Empty)
            {
                string[] values = s.Split(',');
                for (int j = 0; j < numOffsets; j++)
                {
                    switch (orfsType)
                    {
                        case "SWIT":
                            dataLog.measAcpr[dataLog.testIndex, j] = Math.Round(Convert.ToDouble(values[64 + j * 2]), 4);
                            break;

                        case "MOD":
                            dataLog.measAcpr[dataLog.testIndex, j] = Math.Round(Convert.ToDouble(values[4 + j * 2]), 4);
                            break;
                        default:
                            break;
                    }
                }
            }
            else
            {
                for (int j = 0; j < numOffsets; j++)
                {
                    dataLog.measAcpr[dataLog.testIndex, j] = 999;
                    //dcLeakage[dataLog.testIndex] = Math.Round(Convert.ToDouble(values[4]), 4);
                }
            }

            // Log Results
            string[] dataNames = { "ORFS400L", "ORFS400U", "ORFS600L", "ORFS600U", "ORFS Time" };
            string[] dataUnits = { "dBc", "dBc", "dBc", "dBc", "ms" };
            string[] numFormats = { "0.00", "0.00", "0.00", "0.00", "0" };
            double[] dataValues = { dataLog.measAcpr[dataLog.testIndex, 0], dataLog.measAcpr[dataLog.testIndex, 1], 
                                      dataLog.measAcpr[dataLog.testIndex, 2], dataLog.measAcpr[dataLog.testIndex, 3], 
                                      sw1.ElapsedMilliseconds };
            dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
        }

        public void configGsmEvmMeas(string standard, double frequency, double RefLevel)
        {
            if (!useM9300) return;
            AgVisa32.viPrintf(xApp, "INST:SEL GSM\n");
            string cmd = string.Format(":FREQ:CENT {0} Hz\n", frequency);
            AgVisa32.viPrintf(xApp, cmd);
            cmd = string.Format(":POW:RANGE {0} dBm\n", RefLevel + arbRmsValue - getOutputAtten(frequency, 0));
            AgVisa32.viPrintf(xApp, cmd);
            //AgVisa32.viPrintf(xApp, "SENSe:RADio:STANdard:SELect W3GPp\n");
            cmd = string.Format(":RAD:DEV {0}\n", standard);
            AgVisa32.viPrintf(xApp, cmd);
            // Setup commands from MCD Examples
            AgVisa32.viPrintf(xApp, "CONF:PFER\n");
            AgVisa32.viPrintf(xApp, "PFER:AVER:COUN 1\n");
            AgVisa32.viPrintf(xApp, "PFER:AVER:STATE OFF\n");
            if (vsaTriggerSource == AgM9391TriggerEnum.AgM9391TriggerFrontPanelTrigger1)
            {
                AgVisa32.viPrintf(xApp, "PFER:TRIG:SOUR EXT1\n");
            }
            else
            {
                AgVisa32.viPrintf(xApp, "PFER:TRIG:SOUR EXT2\n");
            }

            AgVisa32.viPrintf(xApp, "*WAI\n");
        }

        public void measGsmEvm()
        {
            if (!useM9300) return;
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            string s = "";
            AgVisa32.viPrintf(xApp, "INIT:PFER\n");
            AgVisa32.viPrintf(xApp, "*OPC\n");
            AgVisa32.viPrintf(xApp, "FETC:PFER?\n");
            // Read results as a string and convert
            AgVisa32.viRead(xApp, out s, 800);
            if (s != String.Empty)
            {
                string[] values = s.Split(',');
                dataLog.evm[dataLog.testIndex] = Math.Round(Convert.ToDouble(values[0]), 4);

            }
            else
            {
                dataLog.evm[dataLog.testIndex] = 999;
                dataLog.dcLeakage[dataLog.testIndex] = 999;
            }


            // Log Results
            string[] dataNames = { "Average Phase Error", "EVM Time" };
            string[] dataUnits = { "Degrees", "ms" };
            string[] numFormats = { "0.00", "0" };
            double[] dataValues = { dataLog.evm[dataLog.testIndex], sw1.ElapsedMilliseconds };
            dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);

        }
        private bool firstCgsm = true;

        public void configGsmCgsmMeas(string standard, double frequency, double RefLevel, int numAverages, string type)
        {
            if (!useM9300) return;
            AgVisa32.viPrintf(xApp, "INST:SEL GSM\n");
            // Setup commands from MCD Examples
            AgVisa32.viPrintf(xApp, ":CONF:CGSM:NDEF\n");
            AgVisa32.viPrintf(xApp, ":CGSM:LIST:FORM PFER\n");
            string cmd = string.Format(":FREQ:CENT {0} Hz\n", frequency);
            AgVisa32.viPrintf(xApp, cmd);
            cmd = string.Format(":POW:RANGE {0} dBm\n", RefLevel + arbRmsValue - getOutputAtten(frequency, 0));
            AgVisa32.viPrintf(xApp, cmd);
            cmd = string.Format(":RAD:DEV {0}\n", standard); // BTS or MS
            AgVisa32.viPrintf(xApp, cmd);
            if (numAverages > 1)
            {
                cmd = string.Format(":CGSM:SWE:BURS:NUMB {0}\n", numAverages);
                AgVisa32.viPrintf(xApp, cmd);
                cmd = string.Format(":CGSM:CAPT {0}MS\n", numAverages * 4.615384);
                AgVisa32.viPrintf(xApp, cmd);
                AgVisa32.viPrintf(xApp, "CGSM:SWE:BURS:INT 4.615384MS\n");
            }
            if (vsaTriggerSource == AgM9391TriggerEnum.AgM9391TriggerFrontPanelTrigger1)
            {
                AgVisa32.viPrintf(xApp, ":TRIGger:CGSM:SEQuence:SOURce EXT1\n");
                AgVisa32.viPrintf(xApp, ":TRIGger:EXT1:DELAY -200us\n");
            }
            else
            {
                AgVisa32.viPrintf(xApp, ":TRIGger:CGSM:SEQuence:SOURce EXT2\n");
                AgVisa32.viPrintf(xApp, ":TRIGger:EXT2:DELAY -200us\n");
            }

            AgVisa32.viPrintf(xApp, ":CGSM:DEM 1\n");
            AgVisa32.viPrintf(xApp, ":CGSM:PVT 0\n");
            AgVisa32.viPrintf(xApp, ":CGSM:ZSP 0\n");
            AgVisa32.viPrintf(xApp, ":CGSM:HARM 0\n");
            AgVisa32.viPrintf(xApp, ":CGSM:ORFS 1\n");
            AgVisa32.viPrintf(xApp, ":CGSM:ORFS:TYPE MOD\n");
            //cmd = string.Format(":CGSM:ORFS:TYPE {0}\n", type); // MOD, SWIT, MSW
            //AgVisa32.viPrintf(xApp, cmd);
            AgVisa32.viPrintf(xApp, ":CGSM:FLIS:ORFS:MOD:STAT 1,0,0,0,1,1,0\n");  // Offsets:  100, 200, 250, 400, 600, 800
            AgVisa32.viPrintf(xApp, ":CGSM:ORFS:AVER:FAST ON\n");
        }

        public void measGsmCgsm(double frequency)
        {
            if (!useM9300) return;
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            string s = "";
            if (firstCgsm)
            {
                // Take one measurement and throw away results
                Delay(100);
                s = "";
                AgVisa32.viPrintf(xApp, ":READ:CGSM1?\n");
                // Read results as a string and convert
                AgVisa32.viRead(xApp, out s, 800);
                firstCgsm = false;
            }


            // Setup commands from MCD Examples
            //AgVisa32.viPrintf(xApp, "INST:SEL GSM\n");
            AgVisa32.viPrintf(xApp, ":CONF:CGSM:NDEF\n");
            string cmd = string.Format(":FREQ:CENT {0} Hz\n", frequency);
            AgVisa32.viPrintf(xApp, cmd);
            AgVisa32.viPrintf(xApp, ":READ:CGSM1?\n");
            // Read results as a string and convert
            AgVisa32.viRead(xApp, out s, 800);
            if (s != String.Empty)
            {
                string[] values = s.Split(',');
                dataLog.evm[dataLog.testIndex] = Math.Round(Convert.ToDouble(values[0]), 4);

                for (int j = 0; j < 4; j++)
                {
                    dataLog.measAcpr[dataLog.testIndex, j] = Math.Round(Convert.ToDouble(values[9 + j * 3]), 4);
                }

            }
            else
            {
                dataLog.evm[dataLog.testIndex] = 999;
                dataLog.dcLeakage[dataLog.testIndex] = 999;
            }

            // Log Results
            string[] dataNames = { "Average Phase Error", "ORFS400L", "ORFS400U", "ORFS600L", "ORFS600U", "ORFS Time" };
            string[] dataUnits = { "Degrees", "dBc", "dBc", "dBc", "dBc", "ms" };
            string[] numFormats = { "0.00", "0.00", "0.00", "0.00", "0.00", "0" };
            double[] dataValues = { dataLog.evm[dataLog.testIndex], dataLog.measAcpr[dataLog.testIndex, 0], 
                                      dataLog.measAcpr[dataLog.testIndex, 1], dataLog.measAcpr[dataLog.testIndex, 2], 
                                      dataLog.measAcpr[dataLog.testIndex, 3], sw1.ElapsedMilliseconds };
            dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
        }

        #endregion

        #region EDGE Measurement Routines

        public void configEdgeOrfsMeas(string standard, double frequency, double RefLevel, int numAverages, string type)
        {
            if (!useM9300) return;
            AgVisa32.viPrintf(xApp, "INST:SEL GSM\n");
            string cmd = string.Format(":FREQ:CENT {0} Hz\n", frequency);
            AgVisa32.viPrintf(xApp, cmd);
            cmd = string.Format(":POW:RANGE {0} dBm\n", RefLevel + arbRmsValue - getOutputAtten(frequency, 0));
            AgVisa32.viPrintf(xApp, cmd);
            //AgVisa32.viPrintf(xApp, "SENSe:RADio:STANdard:SELect W3GPp\n");
            cmd = string.Format(":RAD:DEV {0}\n", standard); // BTS or MS
            AgVisa32.viPrintf(xApp, cmd);
            // Setup commands from MCD Examples

            cmd = string.Format(":EORF:AVER:COUN {0}\n", numAverages);
            AgVisa32.viPrintf(xApp, cmd);
            if (numAverages > 1)
                AgVisa32.viPrintf(xApp, ":EORF:AVER:STATE ON\n");
            else
                AgVisa32.viPrintf(xApp, ":EORF:AVER:STATE OFF\n");
            cmd = string.Format(":EORF:TYPE {0}\n", type);
            AgVisa32.viPrintf(xApp, cmd);
            if (vsaTriggerSource == AgM9391TriggerEnum.AgM9391TriggerFrontPanelTrigger1)
            {
                AgVisa32.viPrintf(xApp, ":EORF:TRIG:SOUR EXT1\n");
            }
            else
            {
                AgVisa32.viPrintf(xApp, ":EORF:TRIG:SOUR EXT2\n");
            }
            AgVisa32.viPrintf(xApp, ":EORF:LIST:MOD:STAT 0,0,0,0,1,1\n");
            AgVisa32.viPrintf(xApp, ":EORF:LIST:SWIT:STAT 1,1,1\n");
            AgVisa32.viPrintf(xApp, ":EORF:LIST:SEL CUST\n");
            AgVisa32.viPrintf(xApp, "*WAI\n");
        }

        public void measEdgeOrfs(int numOffsets, string orfsType)
        {
            if (!useM9300) return;
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            string s = "";
            AgVisa32.viPrintf(xApp, "INIT:EORF\n");
            AgVisa32.viPrintf(xApp, "*OPC\n");
            AgVisa32.viPrintf(xApp, "FETC:EORF?\n");
            // Read results as a string and convert
            AgVisa32.viRead(xApp, out s, 8000);
            if (s != String.Empty)
            {
                string[] values = s.Split(',');
                for (int j = 0; j < numOffsets; j++)
                {
                    switch (orfsType)
                    {
                        case "SWIT":
                            dataLog.measAcpr[dataLog.testIndex, j] = Math.Round(Convert.ToDouble(values[64 + j * 2]), 4);
                            break;

                        case "MOD":
                            dataLog.measAcpr[dataLog.testIndex, j] = Math.Round(Convert.ToDouble(values[4 + j * 2]), 4);
                            break;
                        default:
                            break;
                    }
                }
            }
            else
            {
                for (int j = 0; j < numOffsets; j++)
                {
                    dataLog.measAcpr[dataLog.testIndex, j] = 999;
                    //dcLeakage[dataLog.testIndex] = Math.Round(Convert.ToDouble(values[4]), 4);
                }
            }

            // Log Results
            string[] dataNames = { "ORFS400L", "ORFS400U", "ORFS600L", "ORFS600U", "ORFS Time" };
            string[] dataUnits = { "dBc", "dBc", "dBc", "dBc", "ms" };
            string[] numFormats = { "0.00", "0.00", "0.00", "0.00", "0" };
            double[] dataValues = {dataLog.measAcpr[dataLog.testIndex, 0],dataLog.measAcpr[dataLog.testIndex, 1], 
                                     dataLog.measAcpr[dataLog.testIndex, 2],dataLog.measAcpr[dataLog.testIndex, 3], 
                                     sw1.ElapsedMilliseconds };
            dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
        }

        public void configEdgeEvmMeas(string standard, double frequency, double RefLevel)
        {
            if (!useM9300) return;
            AgVisa32.viPrintf(xApp, "INST:SEL GSM\n");
            string cmd = string.Format(":FREQ:CENT {0} Hz\n", frequency);
            AgVisa32.viPrintf(xApp, cmd);
            cmd = string.Format(":POW:RANGE {0} dBm\n", RefLevel + arbRmsValue - getOutputAtten(frequency, 0));
            AgVisa32.viPrintf(xApp, cmd);
            //AgVisa32.viPrintf(xApp, "SENSe:RADio:STANdard:SELect W3GPp\n");
            cmd = string.Format(":RAD:DEV {0}\n", standard);
            AgVisa32.viPrintf(xApp, cmd);
            // Setup commands from MCD Examples
            AgVisa32.viPrintf(xApp, "CONF:EEVM\n");
            AgVisa32.viPrintf(xApp, "EEVM:AVER:COUN 1\n");
            AgVisa32.viPrintf(xApp, "EEVM:AVER:STATE OFF\n");

            if (vsaTriggerSource == AgM9391TriggerEnum.AgM9391TriggerFrontPanelTrigger1)
            {
                AgVisa32.viPrintf(xApp, "EEVM:TRIG:SOUR EXT1\n");
            }
            else
            {
                AgVisa32.viPrintf(xApp, "EEVM:TRIG:SOUR EXT2\n");
            }
            AgVisa32.viPrintf(xApp, "*WAI\n");
        }

        public void measEdgeEvm()
        {
            if (!useM9300) return;
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            string s = "";
            AgVisa32.viPrintf(xApp, "INIT:EEVM\n");
            AgVisa32.viPrintf(xApp, "*OPC\n");
            AgVisa32.viPrintf(xApp, "FETC:EEVM?\n");
            // Read results as a string and convert
            AgVisa32.viRead(xApp, out s, 800);
            if (s != String.Empty)
            {
                string[] values = s.Split(',');
               dataLog.evm[dataLog.testIndex] = Math.Round(Convert.ToDouble(values[1]), 4);

            }
            else
            {
               dataLog.evm[dataLog.testIndex] = 999;
               dataLog.dcLeakage[dataLog.testIndex] = 999;
            }

            // Log Results
            string[] dataNames = { "EVM (RMS)", "EVM Time" };
            string[] dataUnits = { "Percent", "ms" };
            string[] numFormats = { "0.00", "0" };
            double[] dataValues = {dataLog.evm[dataLog.testIndex], sw1.ElapsedMilliseconds };
            dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
        }

        public void configEdgeCgsmMeas(string standard, double frequency, double RefLevel, int numAverages, string type)
        {
            if (!useM9300) return;
            AgVisa32.viPrintf(xApp, "INST:SEL GSM\n");
            // Setup commands from MCD Examples
            AgVisa32.viPrintf(xApp, ":CONF:CGSM:NDEF\n");
            AgVisa32.viPrintf(xApp, ":CGSM:LIST:FORM EEVM\n");
            string cmd = string.Format(":FREQ:CENT {0} Hz\n", frequency);
            AgVisa32.viPrintf(xApp, cmd);
            cmd = string.Format(":POW:RANGE {0} dBm\n", RefLevel + arbRmsValue - getOutputAtten(frequency, 0));
            AgVisa32.viPrintf(xApp, cmd);
            cmd = string.Format(":RAD:DEV {0}\n", standard); // BTS or MS
            AgVisa32.viPrintf(xApp, cmd);
            if (numAverages > 1)
            {
                cmd = string.Format(":CGSM:SWE:BURS:NUMB {0}\n", numAverages);
                AgVisa32.viPrintf(xApp, cmd);
                cmd = string.Format(":CGSM:CAPT {0}MS\n", numAverages * 4.615384);
                AgVisa32.viPrintf(xApp, cmd);
                AgVisa32.viPrintf(xApp, "CGSM:SWE:BURS:INT 4.615384MS\n");
            }
            if (vsaTriggerSource == AgM9391TriggerEnum.AgM9391TriggerFrontPanelTrigger1)
            {
                AgVisa32.viPrintf(xApp, ":TRIGger:CGSM:SEQuence:SOURce EXT1\n");
                AgVisa32.viPrintf(xApp, ":TRIGger:EXT1:DELAY -200us\n");
            }
            else
            {
                AgVisa32.viPrintf(xApp, ":TRIGger:CGSM:SEQuence:SOURce EXT2\n");
                AgVisa32.viPrintf(xApp, ":TRIGger:EXT2:DELAY -200us\n");
            }

            AgVisa32.viPrintf(xApp, ":CGSM:DEM 1\n");
            AgVisa32.viPrintf(xApp, ":CGSM:PVT 0\n");
            AgVisa32.viPrintf(xApp, ":CGSM:ZSP 0\n");
            AgVisa32.viPrintf(xApp, ":CGSM:HARM 0\n");
            AgVisa32.viPrintf(xApp, ":CGSM:ORFS 1\n");
            AgVisa32.viPrintf(xApp, ":CGSM:ORFS:TYPE MOD\n");
            //cmd = string.Format(":CGSM:ORFS:TYPE {0}\n", type); // MOD, SWIT, MSW
            //AgVisa32.viPrintf(xApp, cmd);
            AgVisa32.viPrintf(xApp, ":CGSM:FLIS:ORFS:MOD:STAT 1,0,0,0,1,1,0\n");  // Offsets:  100, 200, 250, 400, 600, 800
            AgVisa32.viPrintf(xApp, ":CGSM:ORFS:AVER:FAST ON\n");
        }

        public void measEdgeCgsm(double frequency)
        {
            if (!useM9300) return;
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            string s = "";
            if (firstCgsm)
            {
                // Take one measurement and throw away results
                Delay(100);
                s = "";
                AgVisa32.viPrintf(xApp, ":READ:CGSM1?\n");
                // Read results as a string and convert
                AgVisa32.viRead(xApp, out s, 800);
                firstCgsm = false;
            }
            // Setup commands from MCD Examples
            //AgVisa32.viPrintf(xApp, "INST:SEL GSM\n");
            AgVisa32.viPrintf(xApp, ":CONF:CGSM:NDEF\n");
            string cmd = string.Format(":FREQ:CENT {0} Hz\n", frequency);
            AgVisa32.viPrintf(xApp, cmd);
            AgVisa32.viPrintf(xApp, ":READ:CGSM1?\n");
            // Read results as a string and convert
            AgVisa32.viRead(xApp, out s, 800);
            if (s != String.Empty)
            {
                string[] values = s.Split(',');
               dataLog.evm[dataLog.testIndex] = Math.Round(Convert.ToDouble(values[1]), 4);

                for (int j = 0; j < 4; j++)
                {
                   dataLog.measAcpr[dataLog.testIndex, j] = Math.Round(Convert.ToDouble(values[14 + j * 3]), 4);
                }

            }
            else
            {
               dataLog.evm[dataLog.testIndex] = 999;
               dataLog.dcLeakage[dataLog.testIndex] = 999;
            }
            // Log Results
            string[] dataNames = { "EVM (RMS)", "ORFS400L", "ORFS400U", "ORFS600L", "ORFS600U", "ORFS Time" };
            string[] dataUnits = { "Degrees", "dBc", "dBc", "dBc", "dBc", "ms" };
            string[] numFormats = { "0.00", "0.00", "0.00", "0.00", "0.00", "0" };
            double[] dataValues = {dataLog.evm[dataLog.testIndex],dataLog.measAcpr[dataLog.testIndex, 0],dataLog.measAcpr[dataLog.testIndex, 1], 
                                     dataLog.measAcpr[dataLog.testIndex, 2],dataLog.measAcpr[dataLog.testIndex, 3], sw1.ElapsedMilliseconds };
            dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
        }

        #endregion

        #region WCDMA Measurement Routines

        public void configWcdmaEvmMeas(string standard, double frequency, double RefLevel)
        {
            if (!useM9300) return;
            AgVisa32.viPrintf(xApp, "INST:SEL WCDMA\n");
            string cmd = string.Format(":FREQ:CENT {0} Hz\n", frequency);
            AgVisa32.viPrintf(xApp, cmd);
            cmd = string.Format(":POW:RANGE {0} dBm\n", RefLevel + arbRmsValue - getOutputAtten(frequency, 0));
            AgVisa32.viPrintf(xApp, cmd);
            //AgVisa32.viPrintf(xApp, "SENSe:RADio:STANdard:SELect W3GPp\n");
            cmd = string.Format(":RAD:DEV {0}\n", standard);
            AgVisa32.viPrintf(xApp, cmd);
            // Setup commands from MCD Examples
            AgVisa32.viPrintf(xApp, "CONF:RHO\n");
            AgVisa32.viPrintf(xApp, "RHO:AVER:COUN 1\n");
            AgVisa32.viPrintf(xApp, "RHO:AVER:STATE OFF\n");
            AgVisa32.viPrintf(xApp, "RHO:CAPT:TIME 0.067\n");
            // Use External Trigger to allow for truncated waveforms
            AgVisa32.viPrintf(xApp, "RHO:TRIG:SOUR EXT2\n");
            AgVisa32.viPrintf(xApp, "*WAI\n");
        }

        public void measWcdmaEvm()
        {
            if (!useM9300) return;
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            string s = "";
            AgVisa32.viPrintf(xApp, "INIT:RHO\n");
            AgVisa32.viPrintf(xApp, "*OPC\n");
            AgVisa32.viPrintf(xApp, "FETC:RHO?\n");
            // Read results as a string and convert
            AgVisa32.viRead(xApp, out s, 800);
            if (s != String.Empty)
            {
                string[] values = s.Split(',');
               dataLog.evm[dataLog.testIndex] = Math.Round(Convert.ToDouble(values[0]), 4);
               dataLog.dcLeakage[dataLog.testIndex] = Math.Round(Convert.ToDouble(values[4]), 4);
            }
            else
            {
               dataLog.evm[dataLog.testIndex] = 999;
               dataLog.dcLeakage[dataLog.testIndex] = 999;
            }

            // Log Results
            string[] dataNames = { "EVM (RMS)", "IQ Origin Offset ", "EVM Time" };
            string[] dataUnits = { "Percent", "dB", "ms" };
            string[] numFormats = { "0.00", "0.00", "0" };
            double[] dataValues = {dataLog.evm[dataLog.testIndex],dataLog.dcLeakage[dataLog.testIndex], sw1.ElapsedMilliseconds };
            dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
        }

        #endregion

        #region TD-SCCDMA Measurement Routines

        public void configTdscdmaEvmMeas(string standard, double frequency, double RefLevel)
        {
            if (!useM9300) return;
            AgVisa32.viPrintf(xApp, "INST:SEL TDSCDMA\n");
            string cmd = string.Format(":FREQ:CENT {0} Hz\n", frequency);
            AgVisa32.viPrintf(xApp, cmd);
            cmd = string.Format(":POW:RANGE {0} dBm\n", RefLevel + arbRmsValue - getOutputAtten(frequency, 0) + refLevelMargin);
            AgVisa32.viPrintf(xApp, cmd);
            cmd = string.Format(":RAD:DEV {0}\n", standard);
            AgVisa32.viPrintf(xApp, cmd);
            // Setup commands from MCD Examples
            AgVisa32.viPrintf(xApp, ":RAD:CONF:HSDP:STAT ON\n");
            AgVisa32.viPrintf(xApp, ":SLOT TS1\n");
            AgVisa32.viPrintf(xApp, ":EVM:AVER:COUN 1\n");
            AgVisa32.viPrintf(xApp, ":EVM:AVER:STATE OFF\n");
            AgVisa32.viPrintf(xApp, ":EVM:RINT 1\n");
            AgVisa32.viPrintf(xApp, ":EVM:TRIG:SOUR IMM\n");
            AgVisa32.viPrintf(xApp, "*WAI\n");
        }

        public void measTdscdmaEvm()
        {
            if (!useM9300) return;
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            string s = "";
            AgVisa32.viPrintf(xApp, "INIT:EVM\n");
            AgVisa32.viPrintf(xApp, "*OPC\n");
            AgVisa32.viPrintf(xApp, "FETC:EVM?\n");
            // Read results as a string and convert
            AgVisa32.viRead(xApp, out s, 800);
            if (s != String.Empty)
            {
                string[] values = s.Split(',');
               dataLog.evm[dataLog.testIndex] = Math.Round(Convert.ToDouble(values[2]), 4);
               dataLog.dcLeakage[dataLog.testIndex] = Math.Round(Convert.ToDouble(values[20]), 4);
            }
            else
            {
               dataLog.evm[dataLog.testIndex] = 999;
               dataLog.dcLeakage[dataLog.testIndex] = 999;
            }

            // Log Results
            string[] dataNames = { "EVM (RMS)", "IQ Origin Offset ", "EVM Time" };
            string[] dataUnits = { "Percent", "dB", "ms" };
            string[] numFormats = { "0.00", "0.00", "0" };
            double[] dataValues = {dataLog.evm[dataLog.testIndex],dataLog.dcLeakage[dataLog.testIndex], sw1.ElapsedMilliseconds };
            dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
        }

        #endregion

        #region CDMA2K/EVDO Measurement Routines

        public void configEvDoRhoMeas(string standard, double frequency, double RefLevel)
        {
            if (!useM9300) return;
            AgVisa32.viPrintf(xApp, "INST:SEL CDMA2K\n");
            string cmd = string.Format(":FREQ:CENT {0} Hz\n", frequency);
            AgVisa32.viPrintf(xApp, cmd);
            cmd = string.Format(":POW:RANGE {0} dBm\n", RefLevel + arbRmsValue - getOutputAtten(frequency, 0));
            AgVisa32.viPrintf(xApp, cmd);
            //AgVisa32.viPrintf(xApp, "SENSe:RADio:STANdard:SELect W3GPp\n");
            cmd = string.Format(":RAD:DEV {0}\n", standard);
            AgVisa32.viPrintf(xApp, cmd);
            // Setup commands from MCD Examples
            AgVisa32.viPrintf(xApp, "CONF:RHO\n");
            AgVisa32.viPrintf(xApp, "RHO:AVER:COUN 1\n");
            AgVisa32.viPrintf(xApp, "RHO:AVER:STATE OFF\n");
            //AgVisa32.viPrintf(xApp, "RHO:CAPT:TIME 0.067\n");
            AgVisa32.viPrintf(xApp, "RHO:TRIG:SOUR IMM\n");
            AgVisa32.viPrintf(xApp, "*WAI\n");
        }

        public void measEvDoRho()
        {
            if (!useM9300) return;
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            string s = "";
            AgVisa32.viPrintf(xApp, "INIT:RHO\n");
            AgVisa32.viPrintf(xApp, "*OPC\n");
            AgVisa32.viPrintf(xApp, "FETC:RHO?\n");
            // Read results as a string and convert
            AgVisa32.viRead(xApp, out s, 800);
            if (s != String.Empty)
            {
                string[] values = s.Split(',');
               dataLog.evm[dataLog.testIndex] = Math.Round(Convert.ToDouble(values[0]), 4);
               dataLog.dcLeakage[dataLog.testIndex] = Math.Round(Convert.ToDouble(values[4]), 4);
            }
            else
            {
               dataLog.evm[dataLog.testIndex] = 999;
               dataLog.dcLeakage[dataLog.testIndex] = 999;
            }

            // Log Results
            string[] dataNames = { "EVM (RMS)", "IQ Origin Offset ", "EVM Time" };
            string[] dataUnits = { "Percent", "dB", "ms" };
            string[] numFormats = { "0.00", "0.00", "0" };
            double[] dataValues = {dataLog.evm[dataLog.testIndex],dataLog.dcLeakage[dataLog.testIndex], sw1.ElapsedMilliseconds };
            dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
        }

        #endregion

        #region LTE - FDD measurement Routines
        public void measLteNS()
        {
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            // Measure Noise Ouput
            Vsa.Arm();
            double measPout = -200;
            bool overload = false;

            Vsa.PowerAcquisition.ReadPower(0, AgM9391PowerUnitsEnum.AgM9391PowerUnitsdBm, ref measPout, ref overload);
            if (overload) throw (new Exception("Overload at measLteNS"));
            measPout += getOutputAtten(Vsa.RF.Frequency + Vsa.PowerAcquisition.OffsetFrequency, 0);
            dataLog.txLeakage[dataLog.testIndex] = measPout;
            long localMeasTime = sw1.ElapsedMilliseconds;

            // Log Results
            string[] dataNames = { "NS Pout", "NS Time" };
            string[] dataUnits = { "dBm", "ms" };
            string[] numFormats = { "0.00", "0" };
            double[] dataValues = { dataLog.txLeakage[dataLog.testIndex], sw1.ElapsedMilliseconds };
            dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
        }

        public void measLteAcpr()
        {
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            double measPout = -200;
            bool overload = false;
            // Measure ACPR 1 and 2 with RRC filter
            Vsa.PowerAcquisition.ChannelFilter.Configure(acprFilterType, FilterAlpha, acprFilterBw);
            Vsa.PowerAcquisition.Duration = acprDuration;
            for (int j = 0; j < 4; j++)
            {
                Vsa.PowerAcquisition.OffsetFrequency = acprOffsetFreq[j];
                Vsa.Apply();
                Vsa.Arm();
                Vsa.PowerAcquisition.ReadPower(0, AgM9391PowerUnitsEnum.AgM9391PowerUnitsdBm,
                        ref measPout, ref overload);
                if (overload) throw (new Exception("Overload at measLteAcpr"));
                measPout += getOutputAtten(Vsa.RF.Frequency + Vsa.PowerAcquisition.OffsetFrequency, 0);
               dataLog.measAcpr[dataLog.testIndex, j] = measPout -dataLog.chanPower[dataLog.testIndex];
               dataLog.measAcprOverload[dataLog.testIndex, j] = overload;
            }
            // Measure ACPR3 with Rectangular filter
            Vsa.PowerAcquisition.ChannelFilter.Configure(FilterType, FilterAlpha, FilterBw);
            Vsa.PowerAcquisition.Duration = acprDuration;
            for (int j = 4; j < 6; j++)
            {
                Vsa.PowerAcquisition.OffsetFrequency = acprOffsetFreq[j];
                Vsa.Apply();
                Vsa.Arm();
                Vsa.PowerAcquisition.ReadPower(0, AgM9391PowerUnitsEnum.AgM9391PowerUnitsdBm,
                        ref measPout, ref overload);
                if (overload) throw (new Exception("Overload at measLteAcpr"));
                measPout += getOutputAtten(Vsa.RF.Frequency + Vsa.PowerAcquisition.OffsetFrequency, 0);
               dataLog.measAcpr[dataLog.testIndex, j] = measPout -dataLog.chanPower[dataLog.testIndex];
               dataLog.measAcprOverload[dataLog.testIndex, j] = overload;
            }
            // Log Results
            string[] dataNames = { "ACPR1L", "ACPR1U", "ACPR2L", "ACPR2U", "ACPR3L", "ACPR3U", "ACPR Time" };
            string[] dataUnits = { "dBc", "dBc", "dBc", "dBc", "dBc", "dBc", "ms" };
            string[] numFormats = { "0.00", "0.00", "0.00", "0.00", "0.00", "0.00", "0" };
            double[] dataValues = {dataLog.measAcpr[dataLog.testIndex, 0],dataLog.measAcpr[dataLog.testIndex, 1], 
                                     dataLog.measAcpr[dataLog.testIndex, 2],dataLog.measAcpr[dataLog.testIndex, 3], 
                                     dataLog.measAcpr[dataLog.testIndex, 4],dataLog.measAcpr[dataLog.testIndex, 5], 
                                     sw1.ElapsedMilliseconds };
            dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
        }

        public void measLteFftAcpr()
        {
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            // For LTE-FDD, RRC filter is used for offsets 1 and 2 and RECT filter is used for offset 3
            // For LTE-TDD RECT filter is used for all offsets, but offsets 1 and 2 are TD-SCDMA channels
            // FilterBw is used for Channel Filter and offset 3 (Always RECT)
            // acprFilterBw is used for offsets 1 and 2 (RRC for FDD, RECT for TDD)

            double fCenter = Vsa.RF.Frequency;
            double[] mChannelPowers = new double[numAcprMeas];
            double[] mChanneldBc = new double[numAcprMeas];
            bool isRrcFilter = false;
            double channelBw = FilterBw;
            double fBinStart;
            int startBin;
            int stopBin;
            double measPout = 0;
            double refPower = 0;
            if (!fftServo)
            {
                // Servo was done with Channel Power measurement, need to make one FFT Acquisition
                // Configure the Acquisition to use Real Time FFT Mode
                Vsa.AcquisitionMode = AgM9391AcquisitionModeEnum.AgM9391AcquisitionModeFFT;
                int spectrumCount = (int)(acprDuration * acprSpan * 1.25 / (int)fftSize);// seconds * samples/second / fftSize
                Vsa.FFTAcquisition.Configure(fftSize, acprSpan * 1.25, spectrumCount, fftWindow, 0);

                // Set the Effective Noise Bandwidth based on Window Type
                // 1.0 for Uniform, 1.5 for Hann, 2.215 for GaussTop, 3.8194 for FlatTop 
                switch (fftWindow)
                {
                    case AgM9391FFTWindowShapeEnum.AgM9391FFTWindowShapeUniform:
                        {
                            windowEnbw = 1.0;
                            break;
                        }
                    case AgM9391FFTWindowShapeEnum.AgM9391FFTWindowShapeHann:
                        {
                            windowEnbw = 1.5;
                            break;
                        }
                    case AgM9391FFTWindowShapeEnum.AgM9391FFTWindowShapeGaussian:
                        {
                            windowEnbw = 2.215;
                            break;
                        }
                    case AgM9391FFTWindowShapeEnum.AgM9391FFTWindowShapeFlatTop:
                        {
                            windowEnbw = 3.8194;
                            break;
                        }
                    default:
                        {
                            throw (new Exception("Window Mode Not Supported:" + fftWindow.ToString()));
                        }
                }
                bool overload = false;
                Vsa.Apply();
                Vsa.Arm();
                Vsa.FFTAcquisition.ReadMagnitudeData(0, ref fftData, ref overload);
                // Switch back to power mode
                Vsa.AcquisitionMode = Agilent.AgM9391.Interop.AgM9391AcquisitionModeEnum.AgM9391AcquisitionModePower;
                Vsa.Apply();

                if (overload) throw (new Exception("Overload at measLteFftAcpr"));
                fCenter = Vsa.RF.Frequency;
                fDelta = acprSpan * 1.25 / (int)fftSize;
                fStart = fCenter - ((int)fftSize / 2 * fDelta) - fDelta / 2;

                // calculate start and stop bin for Reference power, including any fractional bin
                fBinStart = fCenter - channelBw / 2;
                startBin = (int)Math.Floor((fBinStart - fStart) / fDelta);
                stopBin = (int)Math.Floor((fBinStart - fStart + channelBw) / fDelta);
                for (int j = startBin; j <= stopBin; j++)
                {
                    // for RECT filtered channels, scale by percentage of bin inside channel for first and last bin
                    if (j == startBin)
                    {
                        double binScale = ((fBinStart - fStart) / fDelta - startBin);
                        measPout += fftData[j] * binScale;
                    }
                    else if (j == stopBin)
                    {
                        double binScale = (stopBin + 1 - (fBinStart - fStart + channelBw) / fDelta);
                        measPout += fftData[j] * binScale;
                    }
                    else
                        measPout += fftData[j];
                }
                measPout = 10 * Math.Log10(measPout);
                // Use uncorrected reference power so we don't have to correct the power in each adjacent channel
                refPower = measPout;
                // Apply correctoins to measured power to check against channel power measurement
                measPout += getOutputAtten(Vsa.RF.Frequency + Vsa.PowerAcquisition.OffsetFrequency, 0);
                // Correct for the ENBW of FFT Window choice
                measPout -= 10 * Math.Log10(windowEnbw);
            }
            else
            {
                // Use the reference power from the Servo Loop, removing the output Attenuation and window ENBW factors since ACPR is relative
                refPower =dataLog.chanPower[dataLog.testIndex] - getOutputAtten(Vsa.RF.Frequency + Vsa.PowerAcquisition.OffsetFrequency, 0) + 10 * Math.Log10(windowEnbw);
            }

            // Loop over channel offsets to calculate power in each channel
            // Measure ACPR 1 and 2 with ACPR filter (RRC for FDD, RECT for TDD)
            channelBw = acprFilterBw;
            if (FilterType == AgM9391ChannelFilterShapeEnum.AgM9391ChannelFilterShapeRootRaisedCosine)
            {
                isRrcFilter = true;
                // use a wider BW for RRC filter to get the power in the filter skirt
                channelBw = acprFilterBw * 1.3;
            }

            for (int i = 0; i < 4; i++)
            {
                // calculate start and stop bin for offset, including any fractional bin
                fBinStart = fCenter + acprOffsetFreq[i] - channelBw / 2;
                startBin = (int)Math.Floor((fBinStart - fStart) / fDelta);
                stopBin = (int)Math.Floor((fBinStart - fStart + channelBw) / fDelta);
                for (int j = startBin; j <= stopBin; j++)
                {
                    if (isRrcFilter)
                    {
                        double binFreq = (fStart + j * fDelta) - fCenter - acprOffsetFreq[i];
                        double binScale = getRrcFilterWeightSquared(binFreq, FilterBw, FilterAlpha);
                        mChannelPowers[i] += fftData[j] * binScale;
                    }
                    else
                    {
                        // for RECT filtered channels, scale by percentage of bin inside channel for first and last bin
                        if (j == startBin)
                        {
                            double binScale = ((fBinStart - fStart) / fDelta - startBin);
                            mChannelPowers[i] += fftData[j] * binScale;
                        }
                        else if (j == stopBin)
                        {
                            double binScale = (stopBin + 1 - (fBinStart - fStart + channelBw) / fDelta);
                            mChannelPowers[i] += fftData[j] * binScale;
                        }
                        else
                            mChannelPowers[i] += fftData[j];
                    }
                }
                mChannelPowers[i] = 10 * Math.Log10(mChannelPowers[i]);
                mChanneldBc[i] = mChannelPowers[i] - refPower;
               dataLog.measAcpr[dataLog.testIndex, i] = mChanneldBc[i];
            }

            // Measure ACPR 3 with channel (RECT) filter
            for (int i = 4; i < 6; i++)
            {
                // calculate start and stop bin for offset, including any fractional bin
                fBinStart = fCenter + acprOffsetFreq[i] - FilterBw / 2;
                startBin = (int)Math.Floor((fBinStart - fStart) / fDelta);
                stopBin = (int)Math.Floor((fBinStart - fStart + FilterBw) / fDelta);
                for (int j = startBin; j <= stopBin; j++)
                {
                    // for RECT filtered channels, scale by percentage of bin inside channel for first and last bin
                    if (j == startBin)
                    {
                        double binScale = ((fBinStart - fStart) / fDelta - startBin);
                        mChannelPowers[i] += fftData[j] * binScale;
                    }
                    else if (j == stopBin)
                    {
                        double binScale = (stopBin + 1 - (fBinStart - fStart + FilterBw) / fDelta);
                        mChannelPowers[i] += fftData[j] * binScale;
                    }
                    else
                        mChannelPowers[i] += fftData[j];
                }
                mChannelPowers[i] = 10 * Math.Log10(mChannelPowers[i]);
                //mChannelPowers[i] += getOutputAtten(M9391.RF.Frequency + M9391.PowerAcquisition.OffsetFrequency, 0);
                mChanneldBc[i] = mChannelPowers[i] - refPower;
               dataLog.measAcpr[dataLog.testIndex,i] = mChanneldBc[i];
            }

            // Log Results
            if (numAcprMeas == 2)
            {
                string[] dataNames = { "ACPR1L", "ACPR1U", "ACPR Time" };
                string[] dataUnits = { "dBc", "dBc", "ms" };
                string[] numFormats = { "0.00", "0.00", "0" };
                //double[] dataValues = {dataLog.measAcpr[dataLog.testIndex, 0],dataLog.measAcpr[dataLog.testIndex, 1], sw1.ElapsedMilliseconds };
                double[] dataValues = {dataLog.measAcpr[dataLog.testIndex, 0],dataLog.measAcpr[dataLog.testIndex, 1], 
                                          sw1.ElapsedMilliseconds };
                dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
            }
            else if (numAcprMeas == 4)
            {
                string[] dataNames = { "ACPR1L", "ACPR1U", "ACPR2L", "ACPR2U", "ACPR Time" };
                string[] dataUnits = { "dBc", "dBc", "dBc", "dBc", "ms" };
                string[] numFormats = { "0.00", "0.00", "0.00", "0.00", "0" };
                //double[] dataValues = {dataLog.measAcpr[dataLog.testIndex, 0],dataLog.measAcpr[dataLog.testIndex, 1], 
                //                     dataLog.measAcpr[dataLog.testIndex, 2],dataLog.measAcpr[dataLog.testIndex, 3], sw1.ElapsedMilliseconds };
                double[] dataValues = {dataLog.measAcpr[dataLog.testIndex, 0],dataLog.measAcpr[dataLog.testIndex, 1], 
                                     dataLog.measAcpr[dataLog.testIndex, 2],dataLog.measAcpr[dataLog.testIndex, 3], 
                                     sw1.ElapsedMilliseconds };
                dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
            }
            else if (numAcprMeas == 6)
            {
                string[] dataNames = { "ACPR1L", "ACPR1U", "ACPR2L", "ACPR2U", "ACPR3L", "ACPR3U", "ACPR Time" };
                string[] dataUnits = { "dBc", "dBc", "dBc", "dBc", "dBc", "dBc", "ms" };
                string[] numFormats = { "0.00", "0.00", "0.00", "0.00", "0.00", "0.00", "0" };
                double[] dataValues = {dataLog.measAcpr[dataLog.testIndex, 0],dataLog.measAcpr[dataLog.testIndex, 1], 
                                     dataLog.measAcpr[dataLog.testIndex, 2],dataLog.measAcpr[dataLog.testIndex, 3], 
                                     dataLog.measAcpr[dataLog.testIndex, 4],dataLog.measAcpr[dataLog.testIndex, 5], 
                                     sw1.ElapsedMilliseconds };
                //double[] dataValues = { mChanneldBc[0], mChanneldBc[1], 
                //                      mChanneldBc[2], mChanneldBc[3], 
                //                      mChanneldBc[4], mChanneldBc[5], sw1.ElapsedMilliseconds };
                dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
            }
        }

        public void configLteEvmMeas(string standard, string bw, double frequency, double RefLevel)
        {
            if (!useM9300) return;
            AgVisa32.viPrintf(xApp, "INST:SEL LTE\n");
            string cmd = string.Format(":FREQ:CENT {0} Hz\n", frequency);
            AgVisa32.viPrintf(xApp, cmd);
            cmd = string.Format(":POW:RANGE {0} dBm\n", RefLevel + arbRmsValue - getOutputAtten(frequency, 0) + refLevelMargin);
            AgVisa32.viPrintf(xApp, cmd);
            cmd = string.Format("SENSe:RADio:STANdard:DIRection {0}\n", standard);
            AgVisa32.viPrintf(xApp, cmd);
            cmd = string.Format("SENSe:RADio:STANdard:PRESet {0}\n", bw);
            AgVisa32.viPrintf(xApp, cmd);
            // Use the PXI Backplane trigger to trigger at start of waveform
            AgVisa32.viPrintf(xApp, ":TRIG:CEVM:SEQ:SOUR EXT2\n\n");
            //AgVisa32.viPrintf(xApp, "*WAI\n");
            AgVisa32.viPrintf(xApp, "EVM:TIME:RES:LENG 1\n");
            AgVisa32.viPrintf(xApp, "EVM:TIME:ASB SLOT\n");
            AgVisa32.viPrintf(xApp, "EVM:TIME:INT:SLOT 1\n");
            AgVisa32.viPrintf(xApp, "CEVM:EVM:COPY\n");
            AgVisa32.viPrintf(xApp, "CEVM:EQU:TRA RSD\n");
            AgVisa32.viPrintf(xApp, "*WAI\n");
        }

        public void measLteEvm()
        {
            if (!useM9300) return;
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            string s = "";
            AgVisa32.viPrintf(xApp, "INIT:CEVM\n");
            //AgVisa32.viPrintf(xApp, "*OPC\n");
            AgVisa32.viPrintf(xApp, "FETC:CEVM?\n");
            // Read results as a string and convert
            AgVisa32.viRead(xApp, out s, 800);
            if (s != String.Empty)
            {
                string[] values = s.Split(',');
               dataLog.evm[dataLog.testIndex] = Math.Round(Convert.ToDouble(values[0]), 4);
               dataLog.dcLeakage[dataLog.testIndex] = Math.Round(Convert.ToDouble(values[18]), 4);
            }
            else
            {
               dataLog.evm[dataLog.testIndex] = 999;
               dataLog.dcLeakage[dataLog.testIndex] = 999;
            }

            // Log Results
            string[] dataNames = { "EVM (RMS)", "IQ Origin Offset ", "EVM Time" };
            string[] dataUnits = { "Percent", "dB", "ms" };
            string[] numFormats = { "0.00", "0.00", "0" };
            double[] dataValues = {dataLog.evm[dataLog.testIndex],dataLog.dcLeakage[dataLog.testIndex], sw1.ElapsedMilliseconds };
            dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
        }


        public void configLteTddEvmMeas(string standard, string bw, double frequency, double RefLevel)
        {
            if (!useM9300) return;
            AgVisa32.viPrintf(xApp, "INST:SEL LTETDD\n");
            string cmd = string.Format(":FREQ:CENT {0} Hz\n", frequency);
            AgVisa32.viPrintf(xApp, cmd);
            cmd = string.Format(":POW:RANGE {0} dBm\n", RefLevel + arbRmsValue - getOutputAtten(frequency, 0) + refLevelMargin);
            AgVisa32.viPrintf(xApp, cmd);
            cmd = string.Format("SENSe:RADio:STANdard:DIRection {0}\n", standard);
            AgVisa32.viPrintf(xApp, cmd);
            cmd = string.Format("SENSe:RADio:STANdard:PRESet {0}\n", bw);
            AgVisa32.viPrintf(xApp, cmd);
            cmd = string.Format("SENSe:RADio:STANdard:PRESet {0}\n", bw);
            AgVisa32.viPrintf(xApp, ":TRIG:CEVM:SEQ:SOUR EXT2\n\n");
            AgVisa32.viPrintf(xApp, "EVM:TIME:RES:LENG 5\n");
            AgVisa32.viPrintf(xApp, "EVM:TIME:ASB SLOT\n");
            AgVisa32.viPrintf(xApp, "EVM:TIME:INT:SLOT 5\n");
            AgVisa32.viPrintf(xApp, "CEVM:EVM:COPY\n");
            AgVisa32.viPrintf(xApp, "CEVM:EQU:TRA RSD\n");
            AgVisa32.viPrintf(xApp, "*WAI\n");
        }

        public void measLteTddEvm()
        {
            if (!useM9300) return;
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            string s = "";
            AgVisa32.viPrintf(xApp, "INIT:CEVM\n");
            //AgVisa32.viPrintf(xApp, "*OPC\n");
            AgVisa32.viPrintf(xApp, "FETC:CEVM?\n");
            // Read results as a string and convert
            AgVisa32.viRead(xApp, out s, 800);
            if (s != String.Empty)
            {
                string[] values = s.Split(',');
               dataLog.evm[dataLog.testIndex] = Math.Round(Convert.ToDouble(values[0]), 4);
               dataLog.dcLeakage[dataLog.testIndex] = Math.Round(Convert.ToDouble(values[18]), 4);
            }
            else
            {
               dataLog.evm[dataLog.testIndex] = 999;
               dataLog.dcLeakage[dataLog.testIndex] = 999;
            }

            // Log Results
            string[] dataNames = { "EVM (RMS)", "IQ Origin Offset ", "EVM Time" };
            string[] dataUnits = { "Percent", "dB", "ms" };
            string[] numFormats = { "0.00", "0.00", "0" };
            double[] dataValues = {dataLog.evm[dataLog.testIndex],dataLog.dcLeakage[dataLog.testIndex], sw1.ElapsedMilliseconds };
            dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
        }
        #endregion

        #region WLAN Measurement Routines

        public void configWlanMeas(string standard, double frequency, double RefLevel, int evmAvgCount, int semAvgCount)
        {
            if (!useM9300) return;
            AgVisa32.viPrintf(xApp, "INST:SEL WLAN\n");
            string cmd = string.Format(":FREQ:CENT {0} Hz\n", frequency);
            AgVisa32.viPrintf(xApp, cmd);
            cmd = string.Format(":POW:RANGE {0} dBm\n", RefLevel + arbRmsValue - getOutputAtten(frequency, 0) + 2);
            AgVisa32.viPrintf(xApp, cmd);
            cmd = string.Format(":RAD:STAN {0}\n", standard);
            AgVisa32.viPrintf(xApp, cmd);

            bool setFormat = false;
            if (setFormat)
            {
                AgVisa32.viPrintf(xApp, "DEM:AC80:FORM QAM256\n");
                AgVisa32.viPrintf(xApp, "DEM:AC40:FORM QAM256\n");
                AgVisa32.viPrintf(xApp, "DEM:N40:FORM QAM64\n");
                AgVisa32.viPrintf(xApp, "DEM:N20:FORM QAM64\n");
                AgVisa32.viPrintf(xApp, ":EVM:TIME:SLEN 90e-6\n");
                AgVisa32.viPrintf(xApp, ":EVM:TIME:INT 6\n");
                AgVisa32.viPrintf(xApp, ":EVM:TIME:RES:AUTO OFF\n");
                AgVisa32.viPrintf(xApp, ":EVM:TIME:RES:LENG 6\n");
                AgVisa32.viPrintf(xApp, ":EVM:TIME:RES:MAX 6\n");
            }
            AgVisa32.viPrintf(xApp, "SWE:EGAT:DEL 610us\n");
            AgVisa32.viPrintf(xApp, "SWE:EGAT:LENG 75us\n");

            if (vsaTriggerSource == Agilent.AgM9391.Interop.AgM9391TriggerEnum.AgM9391TriggerPXITrigger2)
            {
                AgVisa32.viPrintf(xApp, ":TRIG:EVM:SOUR EXT2\n");
                AgVisa32.viPrintf(xApp, ":TRIG:SEM:SOUR EXT2\n");
            }
            else if (vsaTriggerSource == Agilent.AgM9391.Interop.AgM9391TriggerEnum.AgM9391TriggerFrontPanelTrigger1)
            {
                AgVisa32.viPrintf(xApp, ":TRIG:EVM:SOUR EXT1\n");
                AgVisa32.viPrintf(xApp, ":TRIG:SEM:SOUR EXT1\n");
            }
            else
            {
                AgVisa32.viPrintf(xApp, ":TRIG:EVM:SOUR RFB\n");
                AgVisa32.viPrintf(xApp, ":TRIG:SEM:SOUR RFB\n");
            }

            AgVisa32.viPrintf(xApp, ":EVM:AVER:STAT 1\n");
            cmd = string.Format(":EVM:AVER:COUN {0} \n", evmAvgCount);
            AgVisa32.viPrintf(xApp, cmd);
            AgVisa32.viPrintf(xApp, ":SEM:AVER:STAT 1\n");
            cmd = string.Format(":SEM:AVER:COUN {0} \n", semAvgCount);
            AgVisa32.viPrintf(xApp, cmd);
            AgVisa32.viPrintf(xApp, "*WAI\n");
        }

        public void measWlanEvm()
        {
            if (!useM9300) return;
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            string s = "";
            // Turn off gate for WLAN
            AgVisa32.viPrintf(xApp, "SWE:EGAT Off\n");
            AgVisa32.viPrintf(xApp, "INIT:EVM\n");
            AgVisa32.viPrintf(xApp, "*WAI\n");
            AgVisa32.viPrintf(xApp, "FETC:EVM1?\n");
            // Read results as a string and convert
            AgVisa32.viRead(xApp, out s, 800);
            if (s != String.Empty)
            {
                string[] values = s.Split(',');
               dataLog.evm[dataLog.testIndex] = Math.Round(Convert.ToDouble(values[1]), 4);
               dataLog.symClkErr[dataLog.testIndex] = Math.Round(Convert.ToDouble(values[11]), 4);
               dataLog.dcLeakage[dataLog.testIndex] = Math.Round(Convert.ToDouble(values[13]), 4);
            }
            else
            {
               dataLog.evm[dataLog.testIndex] = 999;
               dataLog.symClkErr[dataLog.testIndex] = 999;
               dataLog.dcLeakage[dataLog.testIndex] = 999;
            }

            // Log Results
            string[] dataNames = { "EVM (RMS)", "Symbol Clock Error", "Carrier Leakage", "EVM Time" };
            string[] dataUnits = { "dB", "ppm", "dB", "ms" };
            string[] numFormats = { "0.00", "0.00", "0.00", "0" };
            double[] dataValues = {dataLog.evm[dataLog.testIndex],dataLog.symClkErr[dataLog.testIndex],
                                      dataLog.dcLeakage[dataLog.testIndex], sw1.ElapsedMilliseconds };
            dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
        }

        public void measWlanSem(int numOffsets)
        {
            if (!useM9300) return;
            //Start Test timer
            sw1.Reset();
            sw1.Start();
            string s = "";
            // turn on gate for SEM
            AgVisa32.viPrintf(xApp, "SWE:EGAT ON\n");
            AgVisa32.viPrintf(xApp, "*WAI\n");
            AgVisa32.viPrintf(xApp, ":READ:SEM?\n");

            // Read results as a string and convert
            AgVisa32.viRead(xApp, out s, 1800);
            if (s != String.Empty)
            {
                string[] values = s.Split(',');
                for (int j = 0; j < numOffsets; j++)
                {
                   dataLog.semMargin[dataLog.testIndex, j] = Math.Round(Convert.ToDouble(values[j + 70]), 4);
                }
            }
            else
            {
                for (int j = 0; j < numOffsets; j++)
                {
                   dataLog.semMargin[dataLog.testIndex, j] = 999;
                }
            }
            long localMeasTime = sw1.ElapsedMilliseconds;

            // Log Results
            string[] dataNames = new string[numOffsets + 1];
            string[] dataUnits = new string[numOffsets + 1];
            string[] numFormats = new string[numOffsets + 1];
            double[] dataValues = new double[numOffsets + 1];
            for (int j = 0; j < numOffsets; j++)
            {
                dataNames[j] = "SEM Margin " + j.ToString();
                dataUnits[j] = "dBc";
                numFormats[j] = "0.00";
                dataValues[j] =dataLog.semMargin[dataLog.testIndex, j];
            }
            dataNames[dataNames.Length - 1] = "SEM Time";
            dataUnits[dataNames.Length - 1] = "ms";
            numFormats[dataNames.Length - 1] = "0";
            dataValues[dataNames.Length - 1] = sw1.ElapsedMilliseconds;
            dataLog.logDataItems(dataNames, dataUnits, numFormats, dataValues);
        }
        #endregion

        #region Simple Example
        private void simpleExample(double centerFreq, double targetPout)
        {
            // Setup the VSA
            Vsa.RF.Frequency = centerFreq;
            Vsa.RF.Power = targetPout;
            Vsa.RF.Conversion = AgM9391ConversionEnum.AgM9391ConversionAuto;
            Vsa.RF.PeakToAverage = arbRmsValue;
            Vsa.RF.IFBandwidth = ifBandwidth;
            // Configure the Acquisition
            Vsa.AcquisitionMode = AgM9391AcquisitionModeEnum.AgM9391AcquisitionModePower;
            Vsa.PowerAcquisition.Bandwidth = sampleRate;
            Vsa.PowerAcquisition.Duration = pwrDuration;
            Vsa.PowerAcquisition.ChannelFilter.Configure(FilterType, FilterAlpha, FilterBw);
            // Send Changes to hardware
            Vsa.Apply();
            Vsa.WaitUntilSettled(100);

            // Set VSA to frequency changing the RF frequency
            string cmd = string.Format(":FREQ:CENT {0} Hz\n", centerFreq);
            AgVisa32.viPrintf(xApp, cmd);
            cmd = string.Format(":POW:RANGE {0} dBm\n", targetPout);
            AgVisa32.viPrintf(xApp, cmd);
            AgVisa32.viPrintf(xApp, ":SENSE:RAD:STAN W3GPP\n");
            cmd = string.Format(":RAD:DEV {0}\n", "MS");
            AgVisa32.viPrintf(xApp, cmd);

            // Measure Power Output
            Vsa.Arm();
            double measPout = -200;
            bool overload = false;
            Vsa.PowerAcquisition.ReadPower(0, AgM9391PowerUnitsEnum.AgM9391PowerUnitsdBm, ref measPout, ref overload);
            if (overload) throw (new Exception("Overload at measStdPout"));

            // Unlock the resource so xApp can use it
            M9000.Checkin(m9000Resource);

            AgVisa32.viPrintf(xApp, "INIT:RHO\n");
            AgVisa32.viPrintf(xApp, "*OPC\n");
            AgVisa32.viPrintf(xApp, "FETC:RHO?\n");
            // Read results as a string and convert
            string s;
            AgVisa32.viRead(xApp, out s, 800);

            // Lock the resource back to the driver
            M9000.Checkout(_key, out m9000Resource);

            // Setup M9391A Driver
            Vsa.RestoreDefaultProperties();
            Vsa.RF.Frequency = centerFreq;
            Vsa.RF.Power = targetPout;
            Vsa.RF.Conversion = AgM9391ConversionEnum.AgM9391ConversionAuto;
            Vsa.RF.PeakToAverage = arbRmsValue;
            Vsa.RF.IFBandwidth = ifBandwidth;
            // Configure the Acquisition
            Vsa.AcquisitionMode = AgM9391AcquisitionModeEnum.AgM9391AcquisitionModePower;
            Vsa.PowerAcquisition.Bandwidth = sampleRate;
            Vsa.PowerAcquisition.Duration = pwrDuration;
            Vsa.PowerAcquisition.ChannelFilter.Configure(FilterType, FilterAlpha, FilterBw);
            // Send Changes to hardware
            Vsa.Apply();
            Vsa.WaitUntilSettled(100);
        }
        #endregion
    }
}